(ns utilities.forms.components
  (:require [hiccup.core :refer [html h]]
            [utilities.exceptions :as ex]))


(defn input-id
  "Creates input id for given field"
  [field]
  (or (:id (:attrs field))
      (str "id_" (:name field))))


(defn label-html
  "returns label for given field"
  [field]
  (html [:label {:for (input-id field)}
         (h (:label field))
         (when (:optional? field) (html [:span.helptext "- Optional"]))
         (when-let [text (:error-msg field)] (html [:div.error text]))
         (when-let [text (:help-msg field)] (html [:br] [:span.helptext text]))]))


(defmulti render
          "Renders html for the given field based on its input type"
          (fn [field value]
            (:field-type field))
          :default :input)

(defmethod render ::select
  [{:keys [name attrs] :as field} value]
  (let [id            (input-id field)
        attrs         (merge {:id id, :name name} attrs)
        attrs         (if (:optional? field)
                        attrs
                        (assoc attrs :required true))
        options-html  (for [[key val] (:options field)]
                        (let [option-attrs  {:value key}
                              option-attrs  (if (= value key)
                                              (assoc option-attrs :selected true)
                                              option-attrs)]
                          (html [:option option-attrs (h val)])))]
    (html [label-html field]
          [:select attrs options-html])))

(defmethod render ::hidden
  [{:keys [name attrs] :as field} value]
  (let [id    (input-id field)
        attrs (merge {:id id, :name name, :type "hidden", :value value}
                     attrs)]
    (html [:input attrs])))

(defmethod render ::checkbox
  [{:keys [name attrs] :as field} value]
  (let [id (input-id field)
        attrs (merge {:id id, :name name}
                     attrs)
        attrs (if value
                (assoc attrs :checked true :value value)
                attrs)]
    (html [:input attrs]
          (label-html field))))

(defmethod render ::textarea
  [{:keys [name attrs] :as field} value]
  (let [id (input-id field)
        attrs (merge {:id id, :name name, :class "u-full-width"}
                     attrs)]
    (html (label-html field)
          [:textarea attrs (h value)])))

(defmethod render ::input
  [{:keys [name attrs] :as field} value]
  (let [id (input-id field)
        attrs (merge {:id id, :type "text", :name name, :value value
                      :class "u-full-width"}
                     attrs)]
    (html (label-html field)
          [:input attrs])))

(defmethod render ::custom-field
  [field value]
  (let [f (:renderer field)]
    (f field value)))

