(ns utilities.forms.core-test
  (:require [clojure.test :refer :all]
            [clojure.data :as d]
            [utilities.forms.core :refer :all]
            [utilities.forms.components :as f]
            [utilities.testing :as testing]
            [utilities.exceptions :as ex]
            [utilities.forms.checks :as v]))


(defn- form-name-is-foo?
  [{:keys [name]}]
  (when-not (= "Foo" name)
    (ex/raise :validation-error "Please check the name.")))

(def name-field (form-field "name" {}))
(def password-field (form-field "password" {}))


(def test-form (form [name-field password-field]
                     [form-name-is-foo?]))


(deftest test-fill-form
  (testing "fill-form"
    (let [person  {:id 1, :name "Harry", :password "bar"}
          form    (fill-form test-form person)]
      (is (= (:data form) {:name "Harry" :password "bar"}))))

  (testing "incremental fill"
    (let [default   {:name "Harry", :password "bar"}
          i         {:password "baz"}
          form      (fill-form test-form default)
          form      (fill-form form i)]
      (is (= (:data form) {:name "Harry" :password "baz"}))))

  (testing "bad incremental fill"
    (let [default   {:id 1, :name "Harry", :password "bar"}
          i         {:password "baz", :id 3}
          form      (assoc test-form :data default)
          form      (fill-form form i)]
      (is (= (:data form) {:id 1 :name "Harry" :password "baz"})))))

(deftest test-render-fields
  (testing "render textarea"
    (let [field (form-field "message" {:field-type ::f/textarea})]
      (is (= (str "<label for=\"id_message\">Message</label>"
                  "<textarea class=\"u-full-width\" id=\"id_message\" name=\"message\"></textarea>")
             (f/render field nil))))))


(deftest test-validate-fields
  (testing "validate email field"
    (let [field  (form-field "email" {:checks [v/email?]})
          result (validate-field field "foo")]
      (is (= (:error-msg result) "Please provide a valid email")
          result)))

  (testing "validate-fields all clean"
    (let [data    {"name" "Foo", "password" "bar"}
          form    (validate-form test-form data)
          fields  (:fields form)]
      (is (= (count fields) 2))
      (is (every? :clean? fields))
      (is (:clean? form))
      (is (= (:error-msg form) nil))
      (is (= (:data form) {:name "Foo", :password "bar"}))))

  (testing "validate-fields without password"
    (let [data      {"name" "Foo"}
          form      (validate-form test-form data)
          fields    (:fields form)
          [name-f password-f] fields]
      (is (= (count fields) 2))
      (is (= (:error-msg form) "Errors in input"))
      (is (not (:clean? form)))
      (is (= (:data form) {:name "Foo", :password nil}))

      (is (= (:error-msg password-f) "This is a required field"))
      (is (not (:clean? password-f)))

      (is (= (:error-msg name-f) nil))
      (is (:clean? name-f))))

  (testing "validate-fields wrong name"
    (let [data {"name" "Wrong", "password" "bar"}
          form (validate-form test-form data)
          fields (:fields form)]
      (is (= (:error-msg form) "Please check the name."))
      (is (not (:clean? form)))
      (is (every? :clean? fields)))))
