(ns utilities.testing
  (:require
    [clojure.tools.namespace.find :as find]
    [clojure.test :refer [run-tests]]
    [utilities.db :as db]
    [utilities.settings :as settings]
    [clojure.java.jdbc :as jdbc]
    [clojure.java.io :as io]
    [clojure.string :as s]))


(defn- run-on-all-tables
  "Runs a command on all tables in database"
  [command]
  (let [tables (jdbc/query (settings/db-spec)
                           ["SELECT table_name
                             FROM information_schema.tables
                             WHERE table_schema = ?" settings/*mysql-db*])
        user-commands (map (fn [{:keys [table_name]}]
                             (str command " " table_name))
                           tables)
        all-commands (concat ["SET FOREIGN_KEY_CHECKS = 0"]
                             user-commands
                             ["SET FOREIGN_KEY_CHECKS=1"])]
    (println "ALL COMMANDS" all-commands)
    (jdbc/db-do-commands (settings/db-spec) all-commands)))


(defn rollback
  "Rolls-back everything by running tests inside a transaction"
  [test]
  (jdbc/with-db-transaction [t-conn (settings/db-spec)]
                            (jdbc/db-set-rollback-only! t-conn)
                            (binding [db/*connection* (delay t-conn)
                                      db/*connection* (delay t-conn)]
                              (test))))


(defn reset-db
  "Drops all tables in database"
  []
  (run-on-all-tables "DROP TABLE IF EXISTS"))


(defn truncate
  "Truncate all tables"
  ([]
   (truncate (fn [] (println "Truncated all data"))))
  ([test]
   (run-on-all-tables "TRUNCATE TABLE")
   (test)))


(defn redirects?
  "Returns true if the response redirects to given location"
  [res loc]
  (let [uri   (get (:headers res) "Location")
        r-loc (str "http://localhost" loc)]
    (or (= loc uri) (= r-loc uri))))


(defn run
  "Parse argument for ns name
  if :reset-db in arguments, then reset-db before test"
  ([] (run ""))
  ([pattern]
   (binding [settings/*in-testing?*     true
             settings/*in-production?*  true
             settings/*mysql-db*        (str "test_" settings/*mysql-db*)]
     ; load urls
     (require '[paper-books.urls])
     ; load tests in given pattern
     (let [ns (find/find-namespaces-in-dir (io/file "test"))
           ns (filter #(s/includes? (str %) pattern) ns)
           ns (map (fn [ns] (require ns) (find-ns ns)) ns)]
       (apply run-tests ns)))))
