(ns utilities.forms.core
  (:require [hiccup.core :refer [html h]]
            [clojure.string :as s]
            [utilities.forms.components :as field]
            [utilities.forms.checks :as v]
            [utilities.exceptions :as ex]))

; Field types

(def input-field ::field/input)
(def custom-field ::field/custom-field)
(def textarea-field ::field/textarea)
(def checkbox-field ::field/checkbox)
(def hidden-field ::field/hidden)
(def select-field ::field/select)

; Map generators

(defn form
  "Create a form map with empty keywords:
  :clean? :error-msg :data"
  ([fields] (form fields nil))
  ([fields checks]
   {:fields    fields
    :checks    checks
    :clean?    nil
    :error-msg nil
    :data      nil}))


(defn form-field
  "Generates a field map with proper defaults"
  ([name] (form-field name {}))

  ([name {:keys [field-type label attrs help-msg optional?
                 checks renderer options]}]

   (when (and (= field-type custom-field)
              (not renderer))
     (throw (Exception. (str "Renderer not provided for custom field " name))))

   (when (and renderer (not= field-type custom-field))
     (throw (Exception. "Renderer is supported only for custom field")))

   (when (and (= field-type select-field)
              (not (map? options)))
     (throw (Exception. (str "No options provided for select field " name))))

   {:field-type    (or field-type input-field)
    :name          name
    :label         (or label (s/capitalize name))
    :attrs         attrs
    :help-msg      help-msg
    :checks        (or checks (when-not optional?
                                (list v/not-blank?)))
    :renderer      renderer
    :options       options
    :clean?        nil
    :error-msg     nil}))


; Renderer

(defn render-form
  "Renders all form fields with errors if any"
  [form]
  (let [form-data   (:data form)
        fields-html (for [field (:fields form)]
                      (field/render field
                                    (get form-data
                                         (keyword (:name field)))))
        msg-html (when-let [msg (:error-msg form)]
                   (html [:div.alert.error (h msg)]))]
    (str msg-html (s/join "\n" fields-html))))


; Validators

(defn- validator-result-field
  "Returns {:clean? true} if all the validators pass on value
  Else returns {:error-msg msg}"
  [checks value]
  (try (every? (fn [check] (check value))
               checks)
       {:clean? true}
       (catch Exception e
         (if (ex/cause? e :validation-error)
           {:error-msg (ex/message e)}
           (throw e)))))


(defn validate-field [field value]
  (merge field
         {:value value}
         (validator-result-field (:checks field) value)))


(defn fill-form
  "Fills fields with the given data values
  Also adds a :data keyword to form map"
  [form data]
  (let [fields        (:fields form)
        keyword-data  (reduce
                        (fn [d {:keys [name]}]
                          (let [kw-name (keyword name)
                                val     (get data name
                                             (get data kw-name
                                                  (get d kw-name)))]
                            (assoc d kw-name val)))
                        (:data form)
                        fields)]
    (assoc form :data keyword-data)))


(defn validate-form
  "First validates all the fields and then runs additional form checks.
  If everything passes, then adds :clean? true,
  else adds :error-msg to the form."
  [form data]
  (let [form              (fill-form form data)
        form-data         (:data form)
        validated-fields   (for [field (:fields form)]
                             (validate-field field (get data (:name field))))
        checks-result (if-not (every? :clean? validated-fields)
                        {:error-msg "Errors in input"}
                        (try
                          (when-let [checks (:checks form)]
                            (every? (fn [check] (check form-data))
                                    checks)
                            {:clean? true})
                          (catch Exception e
                            (if (ex/cause? e :validation-error)
                              {:error-msg (ex/message e)}
                              (throw e)))))]
    (merge form
           {:fields validated-fields}
           checks-result)))


(defn validate-on-post
  "Validates given Form type with the form data if request method is POST"
  [form request]
  (let [post? (= (:request-method request) :post)]
    (if post?
      (validate-form form (:form-params request))
      form)))
