# Generating HTML Content

In Django we have a concept of "extending" templates. So there is base template where we create blocks - and then we have other html files which extends those base template and fills those blocks.

Within templates we have tags and filters. Tags are like `for`, `if`, `cycle`, `groupby`, `firstof`... and filters are like `capfirst`, `join`, `length`, `lower`...this is suddenly a new language inside templating. Also, all of this is very inefficient, so nested `if` and nested `for` are not exactly recommended.

In Clojure, the popular library is `hiccup`. It takes a vector of vectors and converts it to html:

```clojure
(html [:div.primary "Hello World"])

; converts to
; <div class="primary">Hello World</div>

(html [:ul (map (fn [i] [:li i])
                [1 2 3 4])])

; converts to
; <ul>
;     <li>1</li>
;     <li>2</li>
;     <li>3</li>
;     <li>4</li>
; </ul>
```

This is pretty easy and efficient because it provides all the high level Clojure functions to generate HTML. I use it mostly for generating dynamic components such as tables or navigation.

However, Hiccup seems overkill to me for static markup around the web-page. For example, if we are creating a login page, then the form fields are dynamic because they provide error messages and validation. However there is still other static HTML which might be too verbose to write using `hiccup`. For this purpose I create a small library which works as a placeholder filler. So we can create a `login.html` file as:

```html
<h1>User Login</h1>

<form method="post">
    ::csrf-token-
    ::form-fields
    
    <button type="submit">Submit</button>
    
    Forgot password? <a href="/password/forgot/">Click here to reset</a>
</form>

Don't have an account? <a href="/register/">Register Here</a>
```

The `template.core` provides a function `fill-in-content` which will simply replace the `::placeholders` with the passed replacements.

```clojure
(fill-in-content
    "login.html"
    {:form-fields (html [:input {:name "username"}]
                        [:password {:name "password"
                                    :type "password"}])})

; returns the content of login.html as string with
; placeholders replaced with given html
```

The library automatically parses few inbuilt placeholders for `csrf-token-` and `flash-`.

----

## "Extending" templates

The above functionality solves the `tags` and `filters` problem for content html and component html. However, large sites often have different HTML templates which use different CSS stylesheets and JS scripts. We usually create a `base.html` in Django with common css for a consistent look and feel. Then we "extend" other templates from `base.html` for different presentation purposes: `wide.html`, `no-sidebar.html`.

In Clojure, I like to use the "fill-in-the-placeholder" approach here too. So I create my `base.html` as:

```html
<html>
    <head>
    ::title
    
    ::css-links
    </head>
    <body>
        <nav>
            <a href="/">HOME</a>
        </nav>

        <div class="content">
            ::content
        </div>
 
        <footer>Copyright @ 2020</footer>
        
        ::java-scripts
    </body>
</html>
```

I then create "extensions" of different fillers for this `base.html` as:

```clojure
(def wide-css (css-links ["base.css" "wide.css"]))

(def simple-css (css-links ["base.css" "single-column.css"]))
(def simple-js (js-links ["search.js"]))

(defn wide-page
      [title content]
      (fill-in-page "base.html" {:css-links    wide-css
                                 :java-scripts nil
                                 :title        title
                                 :content      content}))
                                 
(defn simple-page
      [title content]
      (fill-in-page "base.html" {:css-links    simple-css
                                 :java-scripts simple-js
                                 :title        title
                                 :content      content}))
```

Now we can use `wide-page` and `simple-page` functions anywhere in the views.

The `js-links` and `css-links` functions take the list of css or js files, and generates hashed links in production. This provides a huge performance boost as we can set very long expiry headers. This optimisation was provided in Django using their `staticfiles` framework.

The `fill-in-page` function is exactly same as `fill-in-content` except that it looks for files in `resources/html` folder while `fill-in-content` looks for files in `resources/html/content` folder.

----

## Performance

- `Hiccup` uses macros which [convert the vectors to html][hiccup-macro] during compilation instead of in run-time.
- The placeholder conversions for content and pages are `memoized` / cached in production. So these are pretty efficient too.

[hiccup-macro]: http://tonsky.me/blog/hiccup/

----

## Why not using Stencil or mustache

**`Mustache` provides good options like quoting, unquoting and functional sections. Then why shouldn't we use Mustache?**

I think that "rich" html components provide lot of assistance such as tooltips, color-highlights, pluralisation, thousand-separtors in numbers etc. These require lots of "(if then else)". Using templating languages will either not provide these or will be very in-efficient with lot of new syntaxes. Placeholders are simple.
