# Rendering forms and Validating form data

We have created a small utility library for rendering and validating forms.

A `form` is map of `fields` and `checks`. We can use the `form` function to generate this map.

Form `fields` are individual items in a form. Each `field` is a map of `name`, `label`, `field-type` (such as input, textarea, checkbox), `attrs` (html attrs passed to the input element) and field level `checks`. We can use the `form-field` function to generate this map. The `field` map must have a name, while all other keywords are optional.

For example, we want to create a user registration form. We would need 3 fields: `email`, `password` and `confirm-password`. The `email` field should check for valid email address and the `password` and `confirm-password` field should be over 6 characters. At total form level, we would also need to check that `password` is equal to `confirm-password`.


## Defining Form Fields

We can generate the `field` map using the `form-field` function. By default, it will make the form-field non-blank unless we set the `:optional?` keyword to `true` in options.

The `field` check functions are called with the field value during validation.

```clojure
(require '[utilities.forms.core as f]
         '[utilities.forms.checks :as v])

; Email field
(f/form-field 'email' {:checks [v/email?]})

; Password field
(f/form-field 'password' {:attrs  {:type "password"}
                          :checks [#(v/min-length? 6 %)]})

; Optional phone-number field
(f/form-field 'phone-number' {:optional? true
                              :checks [v/integer-str?]})

; Textarea field for long message
(f/form-field 'message' {:field-type :textarea
                         :attrs {:cols 60 :rows 5}})

; Select box to select a plan from various plans
(f/form-field 'plan' {:field-type :select
                      :options    {"hobby"    "Hobby (limited usage)"
                                   "personal" "Unlimited for 1 Person"
                                   "family"   "Unlimited for 4"}})

; Checkbox for agreeing terms and conditions
(f/form-field 'agree-terms' {:field-type :checkbox})


; Optional Checkbox for receiving updates
(f/form-field 'agree-terms' {:field-type :checkbox
                             :optional? true
                             :value true})
```

## Defining Forms

The `field` checks checked the field value in standalone. Database level checks or cross-field checks are provided in the `Form` map.

All the form `:checks` are called with the form data during validation.


```clojure
(require '[utilities.forms.core :as f]
         '[utilities.forms.checks :as v])

(def email (f/form-field "email" {:checks [v/email?]}))

(def password (f/form-field "password"
                            {:attrs {:type "password"}
                             :checks [#(v/min-length? 6 %)]}))

(def confirm-password (f/form-field "confirm-password"
                                    {:attrs {:type "password"}
                                     :checks [#(v/min-length? 6 %)]}))

(defn matching-passwords?
      "Returns true if the passwords match in the data"
      [data]
      (if (= (:password data) (:confirm-password data))
          true
          (ex/raise :validation-error "Passwords should be same")))

(def login-form (f/form [email password confirm-password]
                        [v/matching-passwords?]))
```


## Validating a Form

```clojure
(require '[utilities.forms.core :as f])

(f/validate-form login-form {"email"            "foo@bar.com"
                             "password"         "foo@123"
                             "confirm-password" "foo@123"})
; Returns form-map along with :data and :clean?
; {...
;  :data    {:email "foo@bar.com", :password "foo@123", :confirm-password "foo@123"}
;  :clean?  true}


(f/validate-form login-form {"password"         "foo@123"
                             "confirm-password" "foo@bar"})

; Returns form-map along with :data and :error-msg?
; Each field will contain their own errors
; {:fields  [{:name     "email"
              :clean?   nil
              :error-msg  "Please provide a valid email"
              ...}]
;  :data        {:email nil, :password "foo@123", :confirm-password "foo@bar"}
;  :error-msg?  "Errors in input"
;  :clean?      nil}
```


## Rendering a form

```clojure
(f/render-form login-form)
; will render the html with all the fields with their :data values and their error-messages if any
```

## Assisting Functions

```clojure
(f/validate-on-post)
```

**fill-form**: Returns form map along with a `:data` keyword. Form map won't have the `error-msg` as it does not do form-validation.

```clojure
(def form
  (f/fill-form email-form {:to "foo@bar.com", :subject "Foobar"}))

(:data form)
; {:to "foo@bar.com", :subject "Foobar", :message nil}
```
