(ns utilities.templates
  (:require [clojure.string :as s]
            [digest]
            [hiccup.core :refer [html]]
            [hiccup.element :refer [link-to]]
            [ring.util.anti-forgery :refer [anti-forgery-field]]
            [utilities.settings :as settings]
            [utilities.files :as files]
            [clojure.java.io :as io]))


(defn static-link
  "Hashes given file (in public resources) and returns hashed path"
  [f]
  (if settings/in-development?
    f
    (let [[name ext]    (files/name-ext f)
          content       (slurp (io/resource (files/path "public" f)))
          hash          (digest/md5 content)
          new-name      (str name "." (subs hash 0 10) "." ext)
          d             (str "resources/public/hashes/" new-name)]
      (io/make-parents d)
      (spit d content)
      (str "hashes/" new-name))))


(defn css-links
  "Hashes given css files and return html to include"
  [coll-f]
  (s/join "\n"
          (for [f coll-f]
            (str "<link href=\"/static/"
                 (static-link f)
                 "\" rel=\"stylesheet\">"))))


(defn- render-flash
  "Generates HTML for flash map"
  [{:keys [klass msg]}]
  (str "<div class=\"alert " klass "\">"
       msg "</div>"))


(defn- get-placeholder-value
  "Returns placeholder value from given context.
  If placeholder is not in context, then '' is used.
  Inbuilt placeholders are in form of ::placeholder-
  Inbuilt placeholders provided are:
  ::csrf-token-
  ::flash-
  "
  [placeholder context]
  (cond
    (= placeholder :csrf-token-) (anti-forgery-field)
    (= placeholder :flash-) (if-let [flash (:flash (:request context))]
                              (render-flash flash)
                              "")
    :else (let [value (or (get context placeholder) "")]
            (if (coll? value)
              (s/join "\n" value)
              value))))


(defn break-text
  "Splits the text into a vector of strings and keywords.
  Keywords are placeholders"
  [text]
  (let [pattern   #"::([\w-]+)"
        ; splitting to -1 ensures empty last string even if pattern found in end
        parts     (s/split text pattern -1)
        keys      (for [[_ key] (re-seq pattern text)]
                    (keyword key))]
    (concat (interleave parts keys)
            [(last parts)])))


(defn- -break-file
  "Reads given file and breaks it into vector of strings and keywords.
  Optimized for production to read and break only once."
  [f]
  (break-text (slurp (io/resource f))))

(def break-file (settings/memoize-in-prod -break-file))


(defn fill-broken-text
  [broken-text context]
  "Fills text broken into keywords and placeholders with context.
  Returns joined text with filled placeholder values."
  (s/join "" (for [part broken-text]
               (if (keyword? part)
                 (get-placeholder-value part context)
                 part))))


(defn- fill-in-file
  "Fills the placeholders in given file template using context."
  [f context]
  (let [broken-text (break-file f)]
    (fill-broken-text broken-text context)))


(defn fill-in-text
  "Fills the placeholders in given text template using context.
  Placeholders are in form of ::placeholder.
  "
  [text context]
  (-> text
      break-text
      (fill-broken-text context)))


(defn fill-in-page
  "Reads given file in pages folder and fills-in the context.
  The read file is memoized for performance."
  [name context]
  (fill-in-file (str "html/" name)
                context))


(defn fill-in-content
  "Parses given content file with the given context.
  Content files are stored in resources/html/content directory."
  [name context]
  (fill-in-file (str "html/content/" name)
                context))
