(ns utilities.db
    (:require [clojure.java.jdbc :as jdbc]
              [utilities.settings :as settings]
              [clojure.string :as s]
              [utilities.templates :as t]
              [utilities.exceptions :as ex])
    (:import (com.mchange.v2.c3p0 ComboPooledDataSource)))


; Function to BUILD a SELECT query
(defn- to-coll
    "Converts the string to coll with string.
    Returns same if already a collection"
    [x]
    (when x
        (if (coll? x)
            x
            [x])))


(defn- integer
    "Converts given value to number else nil"
    [x]
    (when x
        (Integer. x)))


(defn query
    "Create a SQL query map"
    ([from parts] (query from parts {}))
    ([from {:keys [select where group-by having limit offset]} params]
     (let []
         {:select    (concat (:select from) (to-coll select))
          :from      (get from :from from)
          :where     (concat (:where from) (to-coll where))
          :group-by  (or group-by (:group-by from))
          :having    (or having (:having from))
          :limit     (or (integer limit) (:limit from))
          :offset    (or (integer offset) (:from offset))
          :params    (merge (:params from) params)})))


(defn- order-params
    "Orders map of params in the order of placeholders"
    [params placeholders]
    (reduce (fn [coll key]
                (conj coll (get params key)))
            []
            placeholders))


(defn sql
    "Converts query map to SQL string"
    [query]
    (let [select    (str "SELECT "
                         (let [clauses (seq (:select query))]
                             (if clauses
                                 (s/join ", " clauses)
                                 "*")))
          from      (str "FROM " (:from query))
          where     (when-let [clauses (:where query)]
                        (str "WHERE " (s/join " AND " clauses)))
          group-by  (when-let [clause (:group-by query)]
                        (str "GROUP BY " clause))
          having    (when-let [clause (:having query)]
                        (str "HAVING " clause))
          limit     (when-let [n (:limit query)]
                        (str "LIMIT " n))
          offset    (when-let [n (:offset query)]
                        (str "OFFSET " n))
          order-by  (when-let [clauses (:order-by query)]
                        (str "ORDER BY " (s/join "," clauses)))
          q-str     (s/join "\n" (remove nil?
                                         [select from where group-by
                                          having limit offset order-by]))
          parts     (t/break-text q-str)
          holders   (filter keyword? parts)
          params    (order-params (:params query) holders)
          q-str     (s/join "" (for [part parts]
                                       (if (keyword? part) "?" part)))]
        (cons q-str params)))


; Functions to EXECUTE the query
; Documentation for jdbc api: https://clojure.github.io/java.jdbc/#clojure.java.jdbc/execute!


(defn- get-pool
    "Creates Database connection pool to be used in queries"
    [{:keys [subname user password]}]
    (let [pool (doto (ComboPooledDataSource.)
                   (.setDriverClass "com.mysql.cj.jdbc.Driver")
                   (.setJdbcUrl (str "jdbc:mysql:" subname))
                   (.setUser user)
                   (.setPassword password)
                   ;; expire excess connections after 30 minutes of inactivity:
                   (.setMaxIdleTimeExcessConnections (* 30 60))
                   ;; expire connections after 3 hours of inactivity:
                   (.setMaxIdleTime (* 3 60 60)))]
        {:datasource pool}))


(def ^:dynamic *connection* (delay (get-pool settings/db-spec)))


(defn create
    [table row]
    (try
        (jdbc/insert! @*connection* table row)
        (catch java.sql.SQLIntegrityConstraintViolationException e
            (if (s/includes? (ex/message e) "Duplicate entry")
                (ex/raise :db-duplicate-error "Record already exists")
                (throw e)))))


(defn insert-or-update
    "Performs insert-on-duplicate-update query"
    [table column-names values update-columns]
    (let [update-columns-ref (s/join ", " (for [col update-columns]
                                             (str col " = VALUES(" col ")")))
          q-str (str "INSERT INTO "
                     table " ("
                     (s/join ", " column-names)
                     ") VALUES ("
                     (s/join ", " (repeat (count column-names) "?"))
                     ") ON DUPLICATE KEY UPDATE "
                     update-columns-ref)
          q (cons q-str values)]
        (jdbc/execute! @*connection* q {:multi? true})))


(defn get!
    "Returns first result for the given query.
    Raises error if none or more than one result exist."
    [q]
    (let [result    (jdbc/query @*connection* (sql q))
          n         (count result)]
        (cond
            (= n 1) (first result)
            (> n 1) (ex/raise :db-error "Found Multiple rows")
            (= n 0) (ex/raise :db-error "No results found"))))


(defn get-or-404
    "Raises 404 error if get! fails"
    [q]
    (try
        (get! q)
        (catch Exception e
            (if (ex/cause? e :db-error)
                (ex/raise :404 "Resource not found")
                (throw e)))))


(defn first!
    "Limits and returns first record for given query
    Can return nil"
    [q]
    (let [q     (assoc q :limit 1)]
        (first (jdbc/query @*connection* (sql q)))))


(defn count!
    "Runs MySQL count query for given query"
    [q]
    (let [q     (assoc q :select ["COUNT(*) as __count"])
          res   (jdbc/query @*connection* (sql q))]
        (:__count (first res))))


(defn all!
    "Returns all results for the query"
    [q]
    (jdbc/query @*connection* (sql q)))


(defn page
    "Limits and offsets results based on given page"
    [q page-size page-number]
    (let [results-count     (count! q)
          pages-count       (Math/ceil (/ results-count page-size))
          page-number       (cond
                                (> page-number pages-count) 1
                                (< page-number 1) 1
                                :else page-number)
          offset            (let [n (* (- page-number 1) page-size)]
                                (if (< n 0)
                                    0
                                    n))
          results           (all! (query q {:limit page-size :offset offset}))]
        {:results       results
         :results-count results-count
         :pages-count   pages-count 
         :start         offset
         :page-number   page-number}))