# Querying Databases

We have a small `db.clj` library for querying database.

## Read Queries

The library provides functions for `building` a query and `executing` the created query.

A `query` is a simple map with these keys: `:select`, `:from`, `:where`, `:group-by`, `:having`, `:limit`, `:offset`, `:order-by` and `:params`.

### Building a query

The `query` function only creates the map for sql query. The `sql` function converts the map to sql string.

The benefit of `query` function is that it allows chaining. So we can keep modifying the `query` map before finally sending it to database for execution.

**`query` function**

The first argument to the query function is the `from` part of a sql query. It can include joins. It is mandatory.

The second argument is map of other parts of the query, i.e. `:select`, `:where`, `:group-by`, `:having`, `:limit`, `:offset` and `:order-by`.

The third argument is the map of params.

We use the same placeholder syntax in query building that we use in templates. So params are keyworded using `::param-name` syntax.

```clojure
(query "users user join books book on user.id = book.user_id"
       {:select "book.name"
        :where "user.id = ::user-id"}
       {:user-id user-id})

(query "users user join books book on user.id = book.user_id"
       {:select "user.name, book.name"
        :where "COUNT(book.user_id) > ::min-books"
        :order-by "COUNT(book.user_id)"}
        {:min-books 2})
        
; chaining
(def user-books (query "users user join books book on user.id = book.user_id"
                       {:select "book.name"
                        :where "user.id = ::user-id"
                        :limit 10}
                       {:user-id user-id}))
(sql (query user-books {:select "book.year"
                        :where "book.year > ::min-year"}
                       {:min-year min-year}))
; SELECT book.name, book.year
; FROM users user join books book on user.id=book.user_id
; WHERE user.id = ? AND book.year > ?
; LIMIT ?
; user_id min-year 10
```

### Executing a query

Once the query is created, we can execute it using these helper functions:

```clojure
; gets first result from the query
(first! q)

; assert that query contains 1 result and get it
(get! q)

; like get! but raises 404 if query doesn't return exact 1 result
(get-or-404 q)

; executes count query to return the count of results for the query
(count! q)

; paginates the query by automatically limiting and offsetting the results based on page-size and page-number.
(page q 25 1)

; return all results for the query
(read! q)

```


## Update and Delete Queries

`create`: for creating single row.

```clojure
(create "users" {:name "foobar"
                 :about "I am foo and I can bar"})
```

`insert-or-update`: This is for bulk insertion and updation of data. It uses the very powerful `INSERT INTO ... ON DUPLICATE UPDATE` syntax to create new rows with given values and update old ones where the unique constraint fails.

The first argument is the names of columns corresponding to values. The last argument is the name of columns to update in case the existing row is there.

```clojure
(insert-or-update "books" "name, author, edition"
                  [["Harry Potter", "J K Rowling", "1st"]
                   ["Clojure Cookbook", "Various", "3rd"]]
                  "edition")
; The above query inserts the given rows and updates the "edition" column if the book already exists.
```
