(ns utilities.migrations
    (:require [utilities.settings :as settings]
              [clojure.java.io :as io]
              [clojure.string :as s]
              [utilities.db :as db]
              [clojure.java.jdbc :as jdbc])
    (:import [liquibase Liquibase Contexts]
             (liquibase.resource FileSystemResourceAccessor)
             (liquibase.database DatabaseFactory Database)
             (liquibase.database.jvm JdbcConnection)))


; Docs: http://www.liquibase.org/javadoc/index.html


(def changelog-file "resources/db/changelog.xml")


(defn- liquibase-db
    "Returns a Liquibase Database Instance from given connection"
    [conn]
    (-> (DatabaseFactory/getInstance)
        (.findCorrectDatabaseImplementation
            (JdbcConnection. (:connection conn)))))


(defmacro ->lb
    "Provides instance of Liquibase class and runs given body on it"
    [& body]
    `(jdbc/with-db-connection
        [conn# settings/db-spec]
        (with-open [db# (liquibase-db conn#)]
            (let [lb# (Liquibase. ^String changelog-file
                                  (FileSystemResourceAccessor.)
                                  ^Database db#)]
                (-> lb# ~@body)))))


(defn pending-count
    "Returns count of pending migrations"
    []
    (->lb (.listUnrunChangeSets (Contexts.))
          count))


(defn update-docs
    "Generates Javadoc-like documentation based on current database and change log."
    [docs-path]
    (->lb (.generateDocumentation docs-path)))


(defn validate
    "Validates changelog file"
    []
    (->lb (.validate)))


(defn migrate
    "Updates database to current version."
    []
    (when (> (pending-count) 0)
        (validate)
        (->lb (.update (Contexts.)))))
