(ns inspector.core
  (:require [inspector.impl :as impl]
            [clojure.reflect]
            [clojure.string :as str]
            [clojure.pprint :as pp]))

(defmacro whereami
  "Prints the current stack trace"
  []
  `(println (str/join "\n" (impl/stack-trace))))

(defmacro env
  "Returns the current lexical bindings as a sorted map"
  []
  `(impl/env))

(defn last-inspection
  "Returns the last lexical bindings captured by doinspect"
  []
  @impl/last-inspection)

(defmacro doinspect
  "Executes the forms and prints debug information to *out*"
  [& forms]
  `(impl/doinspect ~@forms))

(defmacro doinspect!
  "Executes the forms and prints debug information to System/out"
  [& forms]
  `(impl/doinspect! ~@forms))

(defn- inspect-form
  "Reader function for doinspect"
  [form]
  `(doinspect ~form))

(defn- inspect-form!
  "Reader function for doinspect!"
  [form]
  `(doinspect! ~form))

(defn ls
  "Prints the members of a namespace, or public members of an Object"
  ([]
   (ls *ns*))
  ([obj]
   (if (instance? clojure.lang.Namespace obj)
     (pp/print-table [:sym :var :doc]
                     (->> (for [[k v] (ns-publics obj)]
                            {:sym k
                             :var v
                             :doc (some-> v meta :doc str/split-lines first)})
                          (sort-by :sym)))
     (pp/print-table [:category :flags :name :parameters :type]
                     (->> obj
                          clojure.reflect/reflect
                          :members
                          (map impl/inspect-member)
                          (filter (every-pred :public? (complement :bridge?)))
                          (sort-by (juxt :category :static? :name :parameters)))))))
