(ns inspector.impl
  (:require [clojure.java.io :as io]
            [clojure.stacktrace :as trace]
            [clojure.string :as str]
            [clojure.walk :as w]
            [puget.printer :as puget]))

(defmacro env
  []
  (into {}
        (for [k (keys &env)
              :when (not (.endsWith (str k) "__auto__"))]
          [(keyword k) (symbol k)])))

(defonce last-inspection (atom nil))
(defonce out-writer (io/writer System/out))
(defonce ^:dynamic *indent* -1)

(defn output
  [& strs]
  (println
   (let [lines (str/split-lines (str/join " " strs))
         lines (map #(str (str/join (repeat *indent* "│ ")) %) lines)]
     (str/join "\n" lines))))

(defn trim-ansi-str
  [s max-width]
  (let [m ^java.util.regex.Matcher (re-matcher #"\u001b\[[0-9;]+m" s)
        ellipsis ".."
        limit (- max-width (count ellipsis))]
    (loop [i 0 w 0]
      (if (.find m)
        (let [code-begin  (.start m)
              code-length (count (.group m))
              str-length  (count (subs s i code-begin))
              new-width   (+ w str-length)
              overflow    (- new-width limit)]
          (if (pos? overflow)
            (str (subs s 0 (max 0 (- code-begin overflow)))
                 "\u001b[m"
                 ellipsis)
            (recur (long (+ code-begin code-length))
                   (long new-width))))
        (let [rest-str (subs s i)
              overflow (- (+ w (count rest-str)) limit)]
          (if (pos? overflow)
            (str (subs s 0 i)
                 (subs rest-str 0 (max 0 (- (count rest-str) overflow)))
                 ellipsis)
            s))))))

(defn ansi-str
  [any]
  (trim-ansi-str (puget/cprint-str any {:width Integer/MAX_VALUE
                                        :seq-limit 100})
                 80))

(defmacro stack-trace
  []
  `(let [x#     (Exception.)
         lines# (str/split-lines (with-out-str (trace/print-stack-trace x#)))]
     (drop 1 lines#)))

(defn trim-form
  "Removes doinspects from the form"
  [form]
  (w/postwalk
   #(if (and (seq? %) (#{'inspector.core/doinspect
                         'inspector.core/doinspect!} (first %)))
      (if (> (count %) 2)
        %
        (last %))
      %)
   form))

(defn wrap-ansi
  [s col]
  (str "\u001b[" col "m" s "\u001b[m"))

(defmacro doinspect
  [& forms]
  `(do
     ~@(for [[idx form] (map vector (iterate inc 0) forms)]
         `(binding [*indent* (inc *indent*)]
            (let [env# (env)
                  len# (apply max 0 (map (comp count name) (keys env#)))
                  form# '~(trim-form form)]
              (when (and (zero? *indent*) (zero? ~idx))
                (output))
              (output (str "┌ " (ansi-str form#)))
              (output (str "│ " (wrap-ansi (first (stack-trace)) "30;1")))
              (let [started# (System/nanoTime)
                    result#  ~form
                    elapsed# (- (System/nanoTime) started#)]
                (reset! last-inspection {:form    form#
                                         :started started#
                                         :result  result#
                                         :elapsed elapsed#
                                         :locals  env#})
                ;; No need to print the evaluation result if not symbol nor list
                (when (or (list? '~form) (symbol? '~form))
                  (output "│  result: " (ansi-str result#))
                  (output "│  elapsed:" (quot elapsed# 1000) "us"))
                ;; Print locals
                (when (seq env#)
                  (output "├  locals:")
                  (doseq [k# (sort (keys env#))
                          :when (not= *out* (env# k#))]
                    (output (format (str "│    %-" (inc len#) "s %s")
                                    (str (name k#) ":")
                                    (ansi-str (env# k#))))))
                (output (str "└ "))
                result#))))))

(defmacro doinspect!
  [& forms]
  `(binding [*out* out-writer]
     (doinspect ~@forms)))

(defn- trim-class-name
  [c]
  (symbol (str/replace-first (name c) #"^java\.lang\." "")))

(defn inspect-member
  [m]
  (let [type (some-> m :type trim-class-name)
        category (cond (:type m) :constant
                       (not (:return-type m)) :constructor
                       :else :method)]
    (assoc m
           :category   category
           :static?    (-> m :flags :static)
           :bridge?    (-> m :flags :bridge)
           :public?    (-> m :flags :public)
           :parameters (some->> m :parameter-types (mapv trim-class-name))
           :type       (or type (some-> m :return-type trim-class-name))
           :flags      (->> (disj (:flags m) :public) (map name) (str/join " ")))))
