(ns jtk-dvlp.re-frame.tasks
  (:require
   [cljs.core.async]
   [jtk-dvlp.async :as a]
   [re-frame.core :as rf]
   [re-frame.interceptor :as interceptor]))


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Functions

(defn register
  [db {:keys [::id] :as task}]
  (assoc-in db [::db :tasks id] task))

(defn unregister
  [db {:keys [::id]}]
  (update-in db [::db :tasks] dissoc id))


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Interceptors

(defn- fx-handler-run?
  [{:keys [stack]}]
  (->> stack
       (filter #(= :fx-handler (:id %)))
       (seq)))

(defn- normalize-task
  [name-or-task]
  (if (map? name-or-task)
    name-or-task
    {::name name-or-task}))

(defn- normalize-fxs
  [fxs]
  (for [fx fxs]
    (cond
      (keyword? fx)
      {:effect-key fx
       :completion-keys #{:on-complete :on-success :on-failure :on-error}}

      (vector? fx)
      {:effect-key (first fx)
       :completion-keys (into #{} (rest fx))}

      :else fx)))

(defonce ^:private !task<->fxs-counters
  (atom {}))

(defn- unregister-by-fx
  [effect completion-keys task]
  (reduce
   (fn [effect completion-key]
     (update effect completion-key (partial vector ::unregister-and-dispatch-original task)))
   effect
   completion-keys))

(defn- unregister-for-fx
  [effects completion-keys task]
  (mapv
   (fn [[effect-key effect-value]]
     [effect-key (unregister-by-fx effect-value completion-keys task)])
   effects))

(defn- unregister-by-fxs
  [context {:keys [::id] :as task} fxs]
  (let [for-fx-count
        (count (rest (:fx (:effects context))))]

    (swap! !task<->fxs-counters assoc id (+ (count fxs) for-fx-count)))

  (reduce
   (fn [context {:keys [effect-key completion-keys]}]
     (if (= effect-key :fx)
       (interceptor/update-effect context effect-key unregister-for-fx completion-keys task)
       (interceptor/update-effect context effect-key unregister-by-fx completion-keys task)))
   context
   fxs))

(defn- unregister-by-failed-acofx
  [context task ?acofx]
  (cljs.core.async/take!
   ?acofx
   (fn [result]
     (when (a/exception? result)
       (rf/dispatch [::unregister task]))))
  context)

(defn- get-db
  [context]
  (or
   (interceptor/get-effect context :db)
   (interceptor/get-coeffect context :db)))

(defn- includes-acofxs?
  [context]
  (contains? context :acoeffects))

(defn- handle-acofx-variant
  [{:keys [acoeffects] :as context} task fxs]
  (let [db
        (get-db context)

        {:keys [dispatch-id ?error]}
        acoeffects

        task
        (assoc task ::id dispatch-id)]

    (if (fx-handler-run? context)
      (if (seq fxs)
        (unregister-by-fxs context task fxs)
        (interceptor/assoc-effect context :db (unregister db task)))
      (-> context
          (interceptor/assoc-effect :db (register db task))
          (unregister-by-failed-acofx task ?error)))))

(defn- handle-straight-variant
  [context task fxs]
  (let [db
        (get-db context)

        task
        (assoc task ::id (random-uuid))]

    ;; NOTE: no need to register task in every case. the task register
    ;;       would be effectiv too late after finish the handler.
    (cond-> context
      (seq fxs)
      (-> (interceptor/assoc-effect :db (register db task))
          (unregister-by-fxs task fxs)))))

(defn as-task
  "Creates an interceptor to mark an event as task.
   Give it a name of the task or map with at least a `::name` key or nil / nothing to use the event name.
   Tasks can be used via subscriptions `::tasks` and `::running?`.

   Given vector `fxs` will be used to identify effects to monitor for the task. Can be the keyword of the effect or an vector of effect keyword and completion keywords to hang in. Completion keys defaults to `:on-complete`, `:on-success`, `on-failure` and `on-error`.

   Given the keyword `all` for `fxs` will hang in for every effect with default completion keys."
  ([]
   (as-task nil))

  ([name-or-task]
   (as-task name-or-task :all))

  ([name-or-task fxs]
   (rf/->interceptor
    :id
    :as-task

    :after
    (fn [context]
      (let [task
            (->> context
                 (:coeffects)
                 (:original-event)
                 (first)
                 (or name-or-task)
                 (normalize-task))

            fx-keys-to-run
            (-> context
                (:effects)
                (keys))

            effective-fx?
            (comp (set fx-keys-to-run) :effect-key)

            effective-fxs
            (if (= fxs :all)
              (normalize-fxs fx-keys-to-run)
              (filter effective-fx? (normalize-fxs fxs)))]

        (cond
          (includes-acofxs? context)
          (handle-acofx-variant context task effective-fxs)

          :else
          (handle-straight-variant context task effective-fxs)))))))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Events

(rf/reg-event-db ::register
  (fn [db [_ task]]
    (register db task)))

(rf/reg-event-db ::unregister
  (fn [db [_ task]]
    (unregister db task)))

(rf/reg-event-fx ::unregister-and-dispatch-original
  (fn [_ [_ task original-event-vec & original-event-args]]
    {::unregister-and-dispatch-original [task original-event-vec original-event-args]}))

(rf/reg-fx ::unregister-and-dispatch-original
  (fn [[{:keys [::id] :as task} original-event-vec original-event-args]]
    (when original-event-vec
      (rf/dispatch (into original-event-vec original-event-args)))

    (if (= 1 (get @!task<->fxs-counters id))
      (do
        (swap! !task<->fxs-counters dissoc id)
        (rf/dispatch [::unregister task]))
      (swap! !task<->fxs-counters update id dec))))


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Subscriptions

(rf/reg-sub ::db
  (fn [{:keys [::db]}]
    db))

(rf/reg-sub ::tasks
  :<- [::db]
  (fn [{:keys [tasks]}]
    tasks))

(rf/reg-sub ::running?
  :<- [::tasks]
  (fn [tasks [_ name]]
    (if name
      (->> tasks (vals) (filter #(= (::name %) name)) (first) (some?))
      (-> tasks (first) (some?)))))
