(ns jp.xcoo.tools.deps-build
  (:require [clojure.edn :as edn]
            [clojure.tools.build.api :as b])
  (:import [java.util ArrayList Properties]
           [org.apache.maven.shared.invoker DefaultInvocationRequest DefaultInvoker]))

(def class-dir "target/classes")

;; delay to defer side effects (artifact downloads)
(def basis (delay (b/create-basis {:project "deps.edn"})))

(defn jar-file
  [lib version]
  (format "target/%s-%s.jar" (name lib) version))

(defn clean [_]
  (b/delete {:path "target"}))

(defn jar
  [{:keys [lib version src-dirs resource-dirs]
    :or {version "LATEST"
         src-dirs ["src"]
         resource-dirs ["resources"]}}]
  (b/write-pom {:class-dir class-dir
                :lib lib
                :version version
                :basis @basis
                :src-dirs src-dirs})
  (b/copy-dir {:src-dirs (concat src-dirs resource-dirs)
               :target-dir class-dir})
  (b/jar {:class-dir class-dir
          :jar-file (jar-file lib version)}))

(defn install
  [{:keys [lib version src-dirs resource-dirs]
    :or {version "LATEST"
         src-dirs ["src"]
         resource-dirs ["resources"]}
    :as args}]
  (clean nil)
  (jar args)
  (b/install {:basis @basis
              :lib lib
              :version version
              :jar-file (jar-file lib version)
              :class-dir class-dir}))

(defn deploy
  [{:keys [lib version repository-id]
    :or {version "LATEST"
         src-dirs ["src"]
         resource-dirs ["resources"]}
    :as args}]
  (clean nil)
  (jar args)
  (let [deps-map (edn/read-string (slurp "deps.edn"))
        request (DefaultInvocationRequest.)
        invoker (DefaultInvoker.)]
    (.setProperties request (doto (Properties.)
                              (.setProperty "repositoryId" repository-id)
                              (.setProperty "url" (get-in deps-map [:mvn/repos repository-id :url]))
                              (.setProperty "file" (.getAbsolutePath (b/resolve-path (jar-file lib version))))))
    (.setGoals request (ArrayList. ["deploy:deploy-file"]))
    (let [result (.execute invoker request)]
      (when-not (= (.getExitCode result) 0)
        (if-let [e (.getExecutionException result)]
          (throw (ex-info "failed to deploy" args e))
          (throw (ex-info "failed to deploy" args)))))))
