(ns i18n.core
  (:import [java.util ResourceBundle$Control])
  (:use clojure.contrib.core
        clojure.contrib.def))

(def ^{:dynamic true :tag String} *default-charset* "UTF-8")

(def in-memory-bundles (atom {}))

(defn to-str [x]
  (cond
    (string? x) x
    (symbol? x) (name x)
    (keyword? x) (name x)))

(def locale-regexp #"([a-zA-Z]+)([-_]([a-zA-Z]+)([-_]([a-zA-Z]+))?)?")

(defn ^java.util.Locale to-locale [x]
  (cond
    (instance? java.util.Locale x) x
    (or (nil? x) (= "" x)) (java.util.Locale. "")
    true
    (if-let [[_ lang _ country _ variant] (re-matches locale-regexp (to-str x))]
      (cond
        variant (java.util.Locale. lang country variant)
        country (java.util.Locale. lang country)
        lang (java.util.Locale. lang)
        true (java.util.Locale. "")))))

(defn- ^java.io.InputStream open-resource
  [resource-name ^ClassLoader loader reload?]
  (if reload?
    (when-let [conn (-?> (.getResource loader resource-name)
                         (.openConnection))]
      (.setUseCaches conn false)
      (.getInputStream conn))
    (.getResourceAsStream loader resource-name)))

(defn get-properties-resource
  [^ResourceBundle$Control rbcontrol base-name locale fmt loader reload?]
  (let [bundle-name (.toBundleName rbcontrol base-name locale)
        resource-name (.toResourceName rbcontrol bundle-name "properties")]
    (when-let [stream (open-resource resource-name loader reload?)]
      (with-open [r (java.io.InputStreamReader. stream *default-charset*)]
        (java.util.PropertyResourceBundle. r)))))

(defn get-clojure-resource [base-name locale]
  (get @in-memory-bundles [base-name locale]))

(defn resource-bundle-control []
  (proxy [ResourceBundle$Control] []
    (getFormats [base-name]
      (cons "clojure.namespace" ResourceBundle$Control/FORMAT_DEFAULT))
    (newBundle [base-name locale fmt loader reload?]
      (cond
        (= fmt "java.properties")
        (get-properties-resource this base-name locale fmt loader reload?)
        (= fmt "clojure.namespace")
        (get-clojure-resource base-name locale)
        :otherwise
        (proxy-super newBundle base-name locale fmt loader reload?)))))

(defn gen-resource [[base-name locale] & resource-map]
  (swap! in-memory-bundles
         assoc [(to-str base-name) (to-locale locale)]
         (proxy [java.util.ListResourceBundle] []
           (getContents []
             (->> (for [[k v] (partition 2 resource-map)]
                    (into-array String [(to-str k) (to-str v)]))
               (into-array (Class/forName "[Ljava.lang.Object;")))))))

(defn-memo ^java.util.ResourceBundle resource-bundle [resource-name locale]
  (java.util.ResourceBundle/getBundle
    (to-str resource-name) (to-locale locale) (clojure.lang.RT/baseLoader)
    (resource-bundle-control)))

(defn-memo resource-keys [resource-name locale]
  (map keyword (enumeration-seq
                 (.getKeys (resource-bundle resource-name locale)))))

(defn-memo resource [resource-name locale resource-key]
  (.getObject (resource-bundle resource-name locale) (to-str resource-key)))

(defn clear-cache []
  (let [cacheList (.getDeclaredField java.util.ResourceBundle "cacheList")]
    (.setAccessible cacheList true)
    (.clear (.get cacheList java.util.ResourceBundle))))

