(ns mdsync.render
  (:require [clojure.java.io :as io]
            [clojure.string :as string]
            [markdown.core :as md]
            [mdsync.blog :as blog]
            [mdsync.core :as core]
            [me.raynes.fs :as fs])
  (:import java.time.LocalDateTime))

(def input-root "resources/page")
(def output-root "target/page")

(defn is-md-file? [file]
  (.endsWith (core/filename file) ".md"))

(defn md-files [root]
  (->> root
       io/file
       file-seq
       (filter is-md-file?)))

(defn render-markdown [content]
  (md/md-to-html-string content))

(defn index-template []
  (slurp (str input-root "/index.html")))

(defn site-destination [filename]
  (str output-root
       "/"
       (core/clean-name filename)
       ".html"))

(comment
  (site-destination "01+allo.md")
  (site-destination "aside.md"))

(def metadata-regex #"^---\n(.+?)\n---")

(defn page-metadata [content]
  (let [metadata (re-find metadata-regex content)]
    (if metadata
      (read-string (second metadata))
      {})))

(comment
  (page-metadata "---\nDINGE\n---\nwullwupp")
  (page-metadata "---\n{:a \"b\"}\n---dingedinge")
  (page-metadata "---\n{:a \"b\"}---dingedinge")
  (page-metadata "aiuaiaia"))

(defn page-content [content]
  (if-let [metadata (re-find metadata-regex content)]
    (subs content (inc (count (first metadata))))
    content))

(comment
  (page-content "---\nDINGE\n---\nwulle")
  (page-content "aiaiaiai"))

(defn now-str []
  (str (LocalDateTime/now)))

(defn file->relpath [file]
  (let [fullpath (-> file .toPath .toString)
        root-pos (string/index-of fullpath input-root)]
    (subs fullpath
          (+ root-pos (count input-root) 1))))

(defn analysed-page [file]
  (let [content (slurp file)
        relpath (file->relpath file)
        blog? (string/starts-with? relpath "blogs/")
        blog-dir (when blog?
                   (-> relpath io/file .toPath
                       (.subpath 1 2) .toString))
        filename (core/clean-name (core/filename file))
        link (if blog?
               (str blog-dir "/" filename ".html")
               (str filename ".html"))]
    {:file file
     :relpath relpath
     :blog? blog?
     :link link
     :filename (core/filename file)
     :content (-> content page-content render-markdown)
     :metadata (page-metadata content)}))

(defn particles-in-content [content]
  (re-seq #"&:particle:([^\s<]+)" content))

(defn last-blog-sites-in-content [content]
  (re-seq #"&:last-blog-sites:([^\s<]+)" content))

(defn envs-in-content [content]
  (re-seq #"&:env:([a-zA-Z_-]+)" content))

(comment
  (particles-in-content "asd &particle:hullu"))

(defn particle-content [particle]
  (-> (str input-root "/particles/" particle ".md")
      io/file
      slurp
      render-markdown))

(defn blog-sites [dir-name]
  (->> (blog/sites-dir dir-name input-root)
       md-files
       (map analysed-page)
       (sort-by :filename)))

(comment
  (particle-content "aside"))

(defn fill-in-placeholders [html page]
  (let [content (:content page "")
        content (string/replace content #"<p><blink></p>" "")
        simple (-> html
                   (string/replace #"&:content" content)
                   (string/replace #"&:title"
                                   (get-in page [:metadata :title] ""))
                   (string/replace #"&:generated-at" (now-str)))
        particles (particles-in-content simple)
        last-blog-sites (last-blog-sites-in-content simple)
        envs (envs-in-content simple)
        particle-fn (fn [result particle]
                      (string/replace
                       result
                       (first particle)
                       (particle-content (second particle))))
        blog-sites-fn (fn [result [match-str blog-name]]
                        (let [sites (blog-sites blog-name)]
                          (string/replace result
                                          match-str
                                          (blog/sites-preview sites))))
        envs-fn (fn [result [match-str env-name]]
                  (string/replace result
                                  match-str
                                  (core/env-val env-name)))]
    (as-> simple $
      (reduce particle-fn $ particles)
      (reduce blog-sites-fn $ last-blog-sites)
      (reduce envs-fn $ envs))))

(comment
  (fill-in-placeholders {:navcode "hhh"}
                        "&:nav &content &:particle:aside\n "
                        {:content "dinge"})
  (fill-in-placeholders {:navcode "hhh"}
                        "&:nav &:content &:particle:aside\n &:generated-at"
                        {:content "dinge"}))

(defn write-sites [template sites]
  (doseq [site sites
          :let [translated (fill-in-placeholders template site)
                filename (core/filename (:file site))
                destination (site-destination filename)]]
    (spit destination translated)))


(defn clear-dir [dir]
  (doseq [file (file-seq dir)]
    (when (.exists dir)
      (when (and (.isDirectory file)
                 (not= dir file))
        (clear-dir file))
      (.delete file))))

(defn blog-dirs []
  (->> (str input-root "/blogs")
       io/file
       .listFiles
       (filter #(.isDirectory %))
       (sort-by #(core/filename %))))

(comment
  (blog-dirs))

(defn render-blog [template dir]
  (let [dir-name (core/filename dir)
        index-file (blog/index-file dir-name input-root)
        config (merge (analysed-page index-file)
                      (blog/config dir-name)
                      {:dir dir})
        blog-sites (blog-sites dir-name)]
    (blog/write-files config
                      output-root
                      blog-sites
                      (partial fill-in-placeholders
                               template))
    config))

(defn copy-resources []
  (fs/copy-dir (str input-root
                    "/resources")
               (str output-root
                    "/resources")))

(defn render
  "Renders all markdown pages under resources/user using the structure in
  resources/user/structure to target/page"
  [project]
  (clear-dir (io/file output-root))
  (io/make-parents (str output-root "/a"))
  (copy-resources)
  (let [sites (md-files (str input-root "/sites"))
        rendered-sites (map analysed-page sites)
        template (index-template)]
    (write-sites template rendered-sites)
    (doseq [blog-dir (blog-dirs)]
      (render-blog template blog-dir))))

(comment
  (render {}))
