(ns mdsync.blog
  (:require [clojure.java.io :as io]
            [clojure.string :as string]
            [mdsync.core :as core]))

(defn index-file [dir-name root]
  (io/file (str root "/blogs/" dir-name "/index.md")))

(defn sites-dir [dir-name root]
  (str root "/blogs/" dir-name "/entries"))

(defn analysed-blog-dir-name [dir-name]
  (let [name-parts (string/split dir-name #"\+")]
    (case (count name-parts)
      1 ["" (first name-parts)]
      2 [(first name-parts) (second name-parts)])))

(comment
  (analysed-blog-dir-name "hudel")
  (analysed-blog-dir-name "4711+wudel"))

(defn config [dir-name]
  (let [[ix blog-dir-name] (analysed-blog-dir-name dir-name)]
    {:ix ix
     :blog-dir-name blog-dir-name}))

(comment
  (config "datblag"))

(defn target-root [config root]
  (str root
       "/"
       (:blog-dir-name config)))

(comment
  (target-root {:blog-dir-name "hudel"}))

(defn site->target-file [config root site]
  (io/file (str (target-root config root)
                "/"
                (core/clean-name (:filename site))
                ".html")))

(defn index-file-content [blog-dir-name sites]
  (str "<ol>"
       (reduce #(str %1
                     "<li>"
                     "<a href=\""
                     (:link %2)
                     "\">"
                     (-> %2 :metadata :title)
                     "</a>"
                     "</li>")
               ""
               sites)
       "</ol>"))

(defn write-files [config output-root sites fill-in-fn]
  (let [index-file (io/file (str (target-root config
                                              output-root)
                                 "/index.html"))]
    (io/make-parents index-file)
    (spit index-file
          (fill-in-fn (assoc config
                             :content
                             (index-file-content
                              (:blog-dir-name config)
                              sites)))))
  (doseq [site sites
          :let [target (site->target-file config output-root site)]]
    (spit target
          (fill-in-fn site))))

(defn sites-preview [sites]
  (reduce (fn [res site]
            (let [content (:content site)
                  divider-pos (string/index-of content "<p><blink></p>")
                  length (count content)
                  end (or divider-pos length)
                  pre-content (subs content 0 end)
                  article-link (str "<a href=\""
                                    (:link site)
                                    "\">"
                                    "↪"
                                    (when divider-pos "…")
                                    "</a>")]
              (str res
                   "<article>"
                   pre-content
                   "<p class=\"article-link\">"
                   article-link
                   "</p>"
                   "</article>")))
          ""
          (take-last
           5
           (reverse sites))))
