(ns formulare.theme
  (:require [clojure.spec.alpha :as spec]
            [hiccup
             [form :as hform]
             [core :as hcore]]))

(defn row [& content] content)

(defn err-msg [content]
  [:section.flash--error
   [:h2.flash__heading--error "Warning"]
   content])

(defn widget-error [id field-def field-value]
  (err-msg
   (map
    (fn [prob]
      [:p
       "Field "
       [:span.flash__field
        (hcore/h (:label field-def))]
       " must comply to "
       [:span.flash__pred
        (hcore/h (:pred prob))]])
    (:clojure.spec.alpha/problems
     (spec/explain-data (:spec field-def)
                        field-value)))))

(defn form-error [form-def req]
  (for [prob (mapcat #(:clojure.spec.alpha/problems
                       (spec/explain-data % req))
                     (:form-specs form-def))]
    (err-msg [:p "The form must comply to "
              (hcore/h (:pred prob))])))

(defn label [field-id field-def]
  (when-not (= :hidden (:widget field-def))
    (hform/label field-id (:label field-def))))

(defn input-widget [id def value]
  (hform/text-field (merge {:required (or (:required def) false)}
                           (:attrs def))
                    id
                    value))

(defn checkbox-widget [id def value]
  (hform/check-box id
                   (and (some? value)
                        (not= 0 value))
                   value))

(defn textarea-widget [id def value]
  (hform/text-area (merge {:required (or (:required def) false)}
                          (:attrs def))
                   id
                   value))

(defn select-widget [id def value]
  (hform/drop-down id
                   (:options def)
                   value))

(defn multiselect-widget [id def value]
  [:select (merge {:multiple "multiple"
                   :size (:size def 5)
                   :name (name id)
                   :id (name id)}
                  (:attrs def))
   (for [option (:options def)]
     [:option {:value (second option)
               :selected (some? (some (partial = (second option))
                                      value))}
      (hcore/h (first option))])])

(comment
  (multiselect-widget :id {:options [["<>" "key"]]} {}))

(defn hidden-widget [id _ value]
  (hform/hidden-field id value))
