(ns formulare.core
  (:require [formulare.theme :as theme]
            [clojure.spec.alpha :as spec]
            [hiccup.form :as hform]
            [hiccup.core :as hcore]
            [ring.util.anti-forgery :refer [anti-forgery-field]]))

(spec/def ::label string?)
(spec/def ::required boolean?)
(spec/def ::spec
  (spec/or :registered-spec keyword?
           :predicate ifn?))
(spec/def ::options
  (spec/or :empty empty?
           :options (spec/coll-of (spec/tuple string? string?))))
(spec/def ::widget
  #{:input :select :checkbox :textarea :mselect :hidden})
(spec/def ::from-req ifn?)
(spec/def ::to-form ::from-req)
(spec/def ::field
  (spec/keys :opt-un [::label
                      ::options
                      ::widget
                      ::from-req
                      ::to-form
                      ::required
                      ::spec]))
(spec/def ::fields
  (spec/map-of keyword? ::field))
(spec/def ::form-specs
  (spec/coll-of ::spec))
(spec/def ::form
  (spec/keys :req-un [::fields]
             :opt-un [::form-specs]))

(defn valid? [form-def req]
  {:pre [(spec/assert ::form form-def)]}
  (reduce-kv
   (fn [result field value]
     (if-let [field-spec (get-in form-def [:fields field :spec])]
       (let [from-req (get-in form-def [:fields field :from-req])
             check-value (if from-req (from-req value) value)]
         (if (spec/valid? field-spec check-value)
           true
           (reduced false)))
       result))
   true
   (:params req)))

(def ^:dynamic *row-theme* theme/row)
(def ^:dynamic *widget-error-theme* theme/widget-error)
(def ^:dynamic *form-error-theme* theme/form-error)
(def ^:dynamic *label-theme* theme/label)
(def ^:dynamic *input-widget-theme* theme/input-widget)
(def ^:dynamic *checkbox-widget-theme* theme/checkbox-widget)
(def ^:dynamic *textarea-widget-theme* theme/textarea-widget)
(def ^:dynamic *select-widget-theme* theme/select-widget)
(def ^:dynamic *mselect-widget-theme* theme/multiselect-widget)
(def ^:dynamic *hidden-widget-theme* theme/hidden-widget)

(defn widget-markup [values req validate? [id def]]
  (let [{:keys [spec widget options to-form]} def
        value ((or to-form identity) (id values))
        req-value (get-in req [:params id])
        renderer (case widget
                   :checkbox *checkbox-widget-theme*
                   :textarea *textarea-widget-theme*
                   :select *select-widget-theme*
                   :mselect *mselect-widget-theme*
                   :hidden *hidden-widget-theme*
                   *input-widget-theme*)]
    (*row-theme* (when (and validate? (not (spec/valid? spec req-value)))
                   (*widget-error-theme* id def req-value))
                 (*label-theme* id def)
                 (renderer id def (if validate? req-value value)))))

(defn form-specs-valid? [form-def req]
  (reduce (fn [valid? field-spec]
            (if (spec/valid? field-spec (:params req))
              true
              (reduced false)))
          true
          (:form-specs form-def)))

(defn form-hash [form-def values]
  (str (hash [form-def values])))

(defn validate? [form-def values req]
  (= (form-hash form-def values)
     (get-in req [:params :__form-hash])))

(defn render-widgets [form-def values req]
  (when-not (spec/valid? ::form form-def)
    (throw (ex-info "Form def fails spec"
                    (spec/explain-data ::form form-def))))
  (let [validate? (validate? form-def values req)
        form-errors (when (and validate?
                               (not (form-specs-valid? form-def req)))
                      (*form-error-theme* form-def req))
        widget-mapper (partial widget-markup values req validate?)
        defined-widgets (map widget-mapper (:fields form-def))
        hash-field (hform/hidden-field "__form-hash"
                                       (form-hash form-def values))
        all-widgets (conj defined-widgets
                          hash-field
                          (anti-forgery-field))]
    (if form-errors
      (concat (if (sequential? form-errors)
                form-errors
                [form-errors])
              all-widgets)
      all-widgets)))

(defn form-data [form-def req]
  (when-not (spec/valid? ::form form-def)
    (throw (ex-info "Form def fails spec" (spec/explain-data ::form form-def))))
  (reduce (fn [coll [id field]]
            (let [value (get-in req [:params id])]
              (assoc coll
                     id
                     (if-let [from-req (:from-req field)]
                       (from-req value)
                       value))))
          {}
          (:fields form-def)))
