(ns joplin.dynamo.utils
  (:require [amazonica.core :refer [parse-args]]
            [amazonica.aws.dynamodbv2 :as dynamo]
            [joplin.core :as joplin]
            [ragtime.protocols :refer [DataStore]]
            [clojure.tools.logging :as log])
  (:import [com.amazonaws.services.dynamodbv2.model ResourceInUseException]
           [com.amazonaws.auth DefaultAWSCredentialsProviderChain]))

(defn wait-for-table-active
  "Waits for # retries for the specified table to become available,
  each retry taking ~1 second."
  ([credentials table-name retries]
   (log/debug "wait-for-table-active" credentials table-name retries)
   (when (>= 0 retries)
     (throw (RuntimeException. (str "Table never became active after " retries " retries."))))
   (let [table-description (if credentials
                             (dynamo/describe-table credentials table-name)
                             (dynamo/describe-table table-name))
         table-status (get-in table-description [:table :table-status])]
     (if (= "ACTIVE" table-status)
       table-description
       (do
         (Thread/sleep 1000)
         (recur credentials table-name (dec retries))))))
  ([table-name retries]
   (wait-for-table-active nil table-name retries)))

;; TODO: Figure out how to make the timout configurable given the variadic nature of amazonica calls.  Probably requires manual testing of last argument

(defn ensure-table
  "Ensures that given Dynamo table exists.
  Takes same arguments as amazonica.aws.dynamodbv2/create-table.
  The implementation will wait up to 360 seconds for the table to become
  available, at which point a RuntimeException is thrown.

  Returns nil if table already exists."
  [& args]
  (let [{parsed-args :args cred :cred} (parse-args (first args) (rest args))
        {table-name :table-name} parsed-args]
    (try
      (if cred
        (do
          (apply (partial dynamo/create-table cred) parsed-args)
          (wait-for-table-active cred table-name 360))
        (do
          (apply dynamo/create-table parsed-args)
          (wait-for-table-active table-name 360)))
      (catch ResourceInUseException exc
        nil))))
