(ns joplin.dynamo.database
  (:require [amazonica.aws.dynamodbv2 :as dynamo]
            [joplin.core :as joplin]
            [joplin.dynamo.utils :refer [ensure-table]]
            [ragtime.protocols :refer [DataStore]])
  (:import [com.amazonaws.services.dynamodbv2.model ResourceInUseException]
           [com.amazonaws.auth DefaultAWSCredentialsProviderChain]))

(def migrations-table-default :migrations)

(defn ensure-migrations-table
  [credentials migrations-table]
  (ensure-table credentials
                :table-name migrations-table
                :key-schema [{:attribute-name :id :key-type :HASH}]
                :attribute-definitions [{:attribute-name :id :attribute-type :S}]
                :provisioned-throughput {:read-capacity-units 1 :write-capacity-units 1}))


;; DYNAMO DATA STORE

(defrecord DynamoDataStore [credentials migrations-table]
  DataStore
  (add-migration-id [_ id]
    ;; Ensure table exists
    (ensure-migrations-table credentials migrations-table)
    ;; Add migration id to table
    (dynamo/put-item credentials
                     :table-name migrations-table
                     :item {:id id :created-at (System/currentTimeMillis)}))

  (remove-migration-id [_ id]
    (ensure-migrations-table credentials migrations-table)
    (dynamo/delete-item credentials
                        :table-name migrations-table
                        :key {:id id}))

  (applied-migration-ids [_]
    (ensure-migrations-table credentials migrations-table)
    (map :id (:items (dynamo/scan credentials :table-name migrations-table)))))

(defn ->dynamo-store
  [target]
  (let [db (:db target)
        credentials (or (:credentials db) (DefaultAWSCredentialsProviderChain.))
        migrations-table (:migrations-table db)]
    (assert migrations-table ":migrations-table is required configuration")
    (->DynamoDataStore credentials migrations-table)))

;; JOPLIN

(defmethod joplin/migrate-db :dynamo [target & args]
  (apply joplin/do-migrate (joplin/get-migrations (:migrator target))
         (->dynamo-store target) args))

(defmethod joplin/rollback-db :dynamo [target amount-or-id & args]
  (apply joplin/do-rollback (joplin/get-migrations (:migrator target))
         (->dynamo-store target) amount-or-id args))

(defmethod joplin/seed-db :dynamo [target & args]
  (apply joplin/do-seed-fn (joplin/get-migrations (:migrator target))
         (->dynamo-store target) target args))

(defmethod joplin/pending-migrations :dynamo [target & args]
  (joplin/do-pending-migrations (->dynamo-store target)
                                (joplin/get-migrations (:migrator target))))

(defmethod joplin/create-migration :dynamo [target id & args]
  (joplin/do-create-migration target id "joplin.dynamo.database"))

