(ns joplin.dynamo.database
  (:require [taoensso.faraday :as dynamo]
            [joplin.core :as joplin]
            [ragtime.core :refer [Migratable]]))

(def migrations-table "migrations")

;; DYNAMO DATA STORE

(defrecord DynamoDataStore [aws-client-opts]
  Migratable
  (add-migration-id [_ id]
    ;; Ensure table exists
    (dynamo/ensure-table aws-client-opts migrations-table [:id :s] {:block? true :throughput {:read 10 :write 10}})
    ;; Add migration id to table
    (dynamo/put-item aws-client-opts migrations-table {:id id :created-at (System/currentTimeMillis)}))

  (remove-migration-id [_ id]
    (dynamo/ensure-table aws-client-opts migrations-table [:id :s] {:block? true :throughput {:read 10 :write 10}})
    (dynamo/delete-item aws-client-opts migrations-table {:id id}))

  (applied-migration-ids [_]
    (dynamo/ensure-table aws-client-opts migrations-table [:id :s] {:block? true :throughput {:read 10 :write 10}})
    (map :id (dynamo/scan aws-client-opts migrations-table))))

(defn ->dynamo-store
  [target]
  (map->DynamoDataStore (select-keys (:db target) [:aws-client-opts])))

;; JOPLIN

(defmethod joplin/migrate-db :dynamo [target & args]
  (apply joplin/do-migrate (joplin/get-migrations (:migrator target))
         (->dynamo-store target) args))

(defmethod joplin/rollback-db :dynamo [target amount-or-id & args]
  (apply joplin/do-rollback (joplin/get-migrations (:migrator target))
         (->dynamo-store target) amount-or-id args))

(defmethod joplin/seed-db :dynamo [target & args]
  (apply joplin/do-seed-fn (joplin/get-migrations (:migrator target))
         (->dynamo-store target) target args))

(defmethod joplin/pending-migrations :dynamo [target & args]
  (joplin/do-pending-migrations (->dynamo-store target)
                                (joplin/get-migrations (:migrator target))))

(defmethod joplin/create-migration :dynamo [target id & args]
  (joplin/do-create-migration target id "joplin.dynamo.database"))

