(ns thingies.db.document
  (:require [com.stuartsierra.component :as component]
            [monger.core :as mongo]
            [monger.collection :as mongo-c]))

;;; INTERFACE
(defprotocol DocumentStore
  (save! [this table record])
  (query [this table query])
  (query-one [this table query])
  (delete! [this table query])
  (update! [this table query record]))

;;; TEST IMPLEMENTATION

(defn entry-match [record [k v]]
  (= (get record k) v))

(defn query-match [q]
  (fn [record]
    (reduce #(and %1 (entry-match record %2)) true q)))

(defn remove-by-query [table query]
  (fn [db]
    (update-in db [table] (partial remove (query-match query)))))

(defrecord TestDB [db-atom]
  DocumentStore
  (save! [this table record]
    (swap! db-atom (fn [db] (update-in db [table] #(conj % record)))))
  (query [this table query]
    (filter (query-match query) (get @db-atom table)))
  (query-one [this table query]
    (throw (Exception. "Implement me")))
  (delete! [this table query]
    (swap! db-atom (remove-by-query table query)))
  (update! [this table query record]
    (swap! db-atom (fn [db] (update-in db [table] (fn [table] (map #(if ((query-match query) %) record %) table)))))))

(defn new-test-db []
  (TestDB. (atom {})))

;;; MONGO IMPLEMENTATION

(defn init-db [this uri]
  (let [{:keys [conn db]} (mongo/connect-via-uri uri)]
    (-> this
        (assoc :connection conn)
        (assoc :db db))))

(defn disconnect [this]
  (mongo/disconnect (:connection this))
  (dissoc this :connection :db))

(defrecord MongoDB [uri]
  component/Lifecycle
  (start [this]
    (println "Starting up MongoDB")
    (init-db this uri))
  (stop [this]
    (println "Stopping MongoDB")
    (disconnect this))
  DocumentStore
  (save! [this table record]
    (mongo-c/save-and-return (:db this) table record))
  (query [this table query]
    (mongo-c/find-maps (:db this) table query))
  (query-one [this table query]
    (throw (Exception. "Implement me")))
  (delete! [this table query]
    (throw (Exception. "Implement me")))
  (update! [this table query record]
    (mongo-c/update (:db this) table query record)))

(defn new-mongo-db [uri]
  (MongoDB. uri))
