(ns jlk.system.core)

;;
;; wrappers for a heap of basic java vm functionality

;;
;; usually to access 'resources', use clojure.java.io/resource
;; note clojure.java.io/resource returns a URL
;;

(defn resource-stream
  "do not using leading / in resource name.  see also clojure.java.io/resource"
  [name]
  (.getResourceAsStream (ClassLoader/getSystemClassLoader) name))

;;
;; runtime
;;

(defn runtime
  "get the current runtime"
  []
  (Runtime/getRuntime))


;;
;; system properties
;;

(defn get-property [key]
  (System/getProperty key))

(defn set-property [key value]
  (System/setProperty key value))

(defn clear-property [key]
  (System/clearProperty key))

(defn user-home
  "user home directory"
  []
  (get-property "user.home"))

(defn user-name
  "user account name"
  []
  (get-property "user.name"))

(defn user-dir
  "user working directory"
  []
  (get-property "user.dir"))

(defn os-arch
  "Operating system architecture"
  []
  (get-property "os.arch"))

(defn os-name
  "Operating system name"
  []
  (get-property "os.name"))

(defn os-version
  "Operating system version"
  []
  (get-property "os.version"))

(defn jvm-version
  ""
  []
  (get-property "java.version"))

(defn jvm-class-path
  ""
  []
  (get-property "java.class.path"))

(defn jvm-library-path
  ""
  []
  (get-property "java.library.path"))

(defn line-separator
  "Sequence used by operating system to separate lines in text files"
  []
  (get-property "line.separator"))

(defn file-separator
  "Character that separates components of a file path. This is \"/\" on UNIX and \"\\\" on Windows."
  []
  (get-property "file.separator"))

;;
;; environment variables
;; 

(defn get-env
  "name as a keyword"
  [key]
  (System/getenv (name key)))

(defn env
  "return the environment as a map, keywords to values"
  []
  (apply merge (map (fn [x] {(keyword (.getKey x)) (.getValue x)}) (System/getenv))))

;;
;; standard streams
;; 

(defn err
  []
  (System/err))

(defn out
  []
  (System/out))

(defn in
  []
  (System/in))

;;
;; query the system
;; 

(defn available-processors
  []
  (.availableProcessors (runtime)))

(defn free-memory
  []
  (.freeMemory (runtime)))

(defn max-memory
  []
  (.maxMemory (runtime)))

(defn total-memory
  []
  (.totalMemory (runtime)))

(defn current-time
  []
  (System/currentTimeMillis))

;;
;; running processes
;; 

;; this is pretty ugly but it works...
(defn exec
  "supply a sequence of strings: command arg1 arg2..., :keys env, wd"
  [cmd & {:keys [env working]}]
  (if (and env working)
    (.exec (runtime)
           (into-array String cmd)
           (into-array String env)
           working)
    (if working
      (.exec (runtime)
             (into-array String cmd)
             nil
             working)
      (if env
        (.exec (runtime)
               (into-array String cmd)
               (into-array String env))
        (.exec (runtime)
               (into-array String cmd))))))

;;
;; running threads
;;

(defn thread
  ([fn]
     (Thread. fn))
  ([kw fn]
     (Thread. (name kw)))
  ([tg kw fn]
     (Thread. tg fn (name kw))))

(defn start
  "do not use .run, use .start instead"
  [t]
  (.start t))

(defn current-thread
  []
  (Thread/currentThread))

(defn current-thread-name
  []
  (.getName (Thread/currentThread)))

(defn current-threadgroup
  []
  (.getThreadGroup (current-thread)))

(defn threadgroup-up
  "search for the threadgroup with name s in the parents of the current threadgroup"
  [s]
  (loop [rv (current-threadgroup)]
    (if (and rv
	     (= (.getName rv) "system"))
      rv
      (if rv
	(recur (.getParent rv))
	(throw (Exception. "thread group not found"))))))

(defn main-threadgroup
  []
  (threadgroup-up "main"))

(defn system-threadgroup
  []
  (threadgroup-up "system"))

(defn list-threads
  "make this list the threads by their group (as a tree)"
  []
  (let [tg (system-threadgroup)
	t (into-array Thread (repeat (.activeCount tg) nil))]
    (.enumerate tg t true)
    (seq t)))

;;
;; vm exit
;; 

(defmulti register-shutdown-hook
  "provide function or thread to run at shutdown"
  class)

(defmethod register-shutdown-hook java.lang.Thread
  [t]
  (.addShutdownHook (runtime) t))

(defmethod register-shutdown-hook java.lang.Runnable
  [fn]
  (.addShutdownHook (runtime) (thread fn)))

(defn remove-shutdown-hook
  "t is a thread"
  [t]
  (.removeShutdownHook (runtime) t))

(defn exit
  "operates on System/exit not Runtime/exit"
  [status]
  (System/exit status))
