(ns jlk.plot.core
  (:use [clojure.test :only [function? deftest run-tests]]
        [jlk.utility.core :only [exception]]
        [seesaw.core :only [frame invoke-later pack! show!]])
  (:import [org.jfree.chart ChartFactory ChartPanel]
           [org.jfree.data.xy XYSeriesCollection XYSeries]
           [org.jfree.chart.plot PlotOrientation]))

(defn -xyseries
  [xydata {:keys [label] :as args}]
  (if-not label (exception "-xyseries: label must be defined"))
  (let [series (XYSeries. label)]
    (doseq [[x y] xydata] (.add series x y))
    series))

(defn -xyplot
  [xydata {:keys [title xaxis yaxis] :as args}]
  (ChartFactory/createXYLineChart title xaxis yaxis
                                  (XYSeriesCollection. (-xyseries xydata args))
                                  PlotOrientation/VERTICAL true true true))

(defn -scatter
  [xydata {:keys [title xaxis yaxis] :as args}]
  (ChartFactory/createScatterPlot title xaxis yaxis
                                  (XYSeriesCollection. (-xyseries xydata args))
                                  PlotOrientation/VERTICAL true true true))

(defn gui
  [chart]
  (let [frame (frame :content (ChartPanel. chart))]
    (invoke-later
     (-> frame pack! show!))))

(defn -range
  [{:keys [range min max step]}]
  (if (and (not range)
           (and (not min)
                (not max)
                (not step)))
    (exception "must define either a range or a min/max/step"))
  (if range
    (apply clojure.core/range range)
    (clojure.core/range min max step)))

(defn plot
  [type this & args]
  (let [plotfn (type {:xy -xyplot
                      :scatter -scatter})
        data (if (function? this)
               (map (fn [x] [x (this x)]) (-range args))
               this)]
   (gui (plotfn data args))))

(deftest -xyplot-test
  (plot :xy
        [[0 0] [100 500] [120 230]]
        :label "data label"
        :title "xy plot"
        :xaxis "xaxis"
        :yaxis "yaxis")
  (plot :scatter
        [[0 0] [100 500] [120 230]]
        :label "data label"
        :title "scatter plot")
  (plot :xy
        (fn [x] (* x x))
        :label "data label"
        :min 0
        :max 10
        :step 0.1))
