(ns jlk.fs.core
  (:refer-clojure :exclude [type time name list])
  (:use [jlk.utility.core :only [exception]]
        [jlk.log.core :only [fatal error warn info debug trace]]
        [jlk.time.core :only [time]]
        [jlk.system.core :only [user-home]])
  (:require [clojure.java.io :as io])
  (:import [org.apache.commons.io FileUtils]))

;;
;; this is similar in intent to fs.core, but implemented differently
;; focus is on programmatic utility
;; concepts of cwd etc. will be implemented in a separate jlk.shell namespace
;;
;; look into this again for Java 7
;;

(defn existing? [f] (.exists f))
(defn file? [f] (.isFile f))
(defn directory? [f] (.isDirectory f))
(defn read? [f] (.canRead f))
(defn hidden? [f] "true for files beginning with '.' on linux" (.isHidden f))
(defn visible? [f] "not hidden?" (not (hidden? f)))
(defn write? [f] (.canWrite f))
(defn execute? [f] (.canExecute f))
(defn absolute? [f] (.isAbsolute f))

(defn type
  "f(ile), d(irectory), n(ew), ? (unknown)"
  [f]
  (if (file? f)
    "f"
    (if (directory? f)
      "d"
      (if (not (existing? f))
        "n"
        "?"))))

(defn permissions
  "get file permissions - only returns the current user permissions

r(ead) -
w(write) -
x(execute) -
"
  [f]
  (str (if (read? f) "r" "-")
       (if (write? f) "w" "-")
       (if (execute? f) "x" "-")))

;; this needs some work...
(defn permissions!
  "set file permissions

r(ead) -
w(write) -
x(execute) -
a(nyone)"
  [f p]
  (let [owner-only (not (get p 3))]
    (if (= (get p 0) \r)
      (.setReadable f true owner-only)
      (if (= (get p 0) \-)
        (.setReadable f false owner-only)))
    (if (= (get p 1) \w)
      (.setWritable f true owner-only)
      (if (= (get p 1) \-)
        (.setWritable f false owner-only)))
    (if (= (get p 2) \x)
      (.setExecutable f true owner-only)
      (if (= (get p 2) \-)
        (.setExecutable f false owner-only))))
  nil)

;; check that this handles ~/ characters correctly.  otherwise pull in from jlk.system/home
(defn file
  "permissions is a string of tests that are performed on the file, otherwise an exception is thrown.  you do not need to provide a full string.

first character is either . (ignore) f(ile) d(irectory) n(ew) ? (unknown)
next three characters are either . (ignore) r(ead) w(rite) x(execute) - (no)

~/ will be translated into (jlk.system/user-home)

eg.
 (file /tmp d) tests that this is a directory
 (file /tmp d.w.) tests that this is a directory we can write to
 (file /tmp/afile frw-) tests that this is a file we can read and write but not execute
"
  ([f] (if (and (instance? String f)
                (.startsWith f "~/"))
         (io/file (str (user-home) (.substring f 1)))
         (io/file f)))
  ([f perm]
     (let [f (file f)
           p (str (type f)
                  (permissions f))]
       (doseq [[x y] (partition 2 (interleave perm p))]
         (if (or (= x y)
                 (= x \.))
           nil
           (exception "file %s: %s != %s" f p perm)))
       f)))

(defn child
  "create a file parent/child"
  ([parent child]
     (if (and (instance? String parent)
              (.startsWith parent "~/"))
       (io/file (str (user-home) (.substring parent 1)) child)
       (io/file parent child)))
  ([p c perm]
     (file (child p c) perm)))

(defn home
  "users home directory"
  []
  (file (user-home)))

(defn create-file!
  [f]
  (if-not (.createNewFile f)
    (exception "file %s: already exists" f)))

(defn delete-file!
  [f]
  (if-not (= (type f) "f")
    (exception "file %s: could not be deleted (type %s != f)" f (type f)))
  (if-not (.delete f)
    (exception "file %s: could not be deleted" f)))

(defn rename!
  [from to]
  (if (= (type from) "n")
    (exception "files %s %s: rename - source file does not exist" from to))
  (if-not (= (type to) "n")
    (exception "files %s %s: rename - destination file already exists" from to))
  (if-not (.renameTo from to)
    (exception "files %s %s: cannot rename" from to)))

;; expand this to recursive delete
(defn delete-directory!
  [f]
  (if-not (= (type f) "d")
    (exception "file %s: could not be deleted (type %s != d)" f (type f)))
  (if-not (.delete f)
    (exception "file %s: could not be deleted" f)))

(defn create-directory!
  [f]
  (if-not (= (type f) "n")
    (exception "file %s: directory could not be created (type %s != n)" f (type f)))
  (if-not (.mkdir f)
    (exception "file %s: directory could not be created" f)))

(defn create-directory-structure!
  [f]
  (if-not (= (type f) "n")
    (exception "file %s: directory structure could not be created (type %s != n)" f (type f)))
  (if-not (.mkdirs f)
    (exception "file %s: directory structure could not be created" f)))

(defn parent [f] (.getParentFile f))
(defn absolute [f] (.getAbsoluteFile f))
(defn canonical [f] (.getCanonicalFile f))


(defn name [f] (.getName f))
(defn path [f] (.getPath f))
(defn parent-path [f] (.getParent f))
(defn absolute-path [f] (.getAbsolutePath f))
(defn canonical-path [f] (.getCanonicalPath f))

(defn split
  "return [name extension]"
  [f]
  (if (= (type f) "d")
    (exception "file %s: is a directory" f))
  (if-let [rv (re-find #"(.*)(\.[A-z0-9]+)$" (name f))]
    (rest rv)
    (clojure.core/list (name f) "")))

(defn file-name
  "return the name of the file without the extension"
  [f]
  (if (= (type f) "d")
    (exception "file %s: is a directory" f))
  (if-let [rv (re-find #"(.*)\.[A-z0-9]+$" (name f))]
    (second rv)
    (name f)))

(defn extension
  "get file extension, or empty string otherwise.  raise exception if given a directory."
  [f]
  (if (= (type f) "d")
    (exception "file %s: is a directory" f))
  (if-let [rv (re-find #"\.[A-z0-9]+$" (name f))]
    rv
    ""))

;; (defn ctime
;;   "file creation time"
;;   [f]
;;   (throw (Exception. "TODO")))

(defn mtime
  "file modification time as joda time"
  [f]
  (time (.lastModified f)))

(defn mlong
  "file modification time as long"
  [f]
  (.lastModified f))

(defn checksum-crc
  "see also: jlk.fs.digest"
  [file]
  (.getValue (FileUtils/checksum file (java.util.zip.CRC32.))))

(defn checksum-adler
  "see also: jlk.fs.digest"
  [file]
  (.getValue (FileUtils/checksum file (java.util.zip.Adler32.))))

(defn newer?
  "check file is newer than x, which may be a file, long or date"
  [file x]
  (FileUtils/isFileNewer file x))

(defn older?
  "check file is older than x, which may be a file, long or date"
  [file x]
  (FileUtils/isFileOlder file x))

(defn size
  "return file length in bytes.  note the size of directories is inconsistent across different file systems and using a directory will raise an exception"
  [f]
  (if (file? f)
    (.length f)
    (exception "file %s: is not a file" f)))

(defn list
  [f & {:keys [filter recursive]}]
  (if-not (= (type f) "d")
    (exception "file %s: cannot list non-directories" f))
  (let [list (if recursive
               (file-seq f)
               (seq (.listFiles f)))]
    (if filter
      (clojure.core/filter filter list)
      list)))
