(ns whitespace-linter.lint
  (:import java.io.File
           java.io.Reader)
  (:require [clojure.java.io :as io]
            [clojure.string :as str]))

(def ^:const max-line-width
  "This is a universal constant, right?"
  80)

(def validators
  #{{:validator/desc "Long lines"
     :validator/name :validator.name/long-lines
     :validator/target :target/each-line
     :validator/valid? #(< (count %) 80)}

    {:validator/desc "Trailing whitespace"
     :validator/name :validator.name/trailing-whitespace
     :validator/target :target/each-line
     :validator/valid? #(not (re-matches #"\s+$" %))}

    {:validator/desc "Missing final newline"
     :validator/name :validator.name/missing-final-newline
     :validator/target :target/all-lines
     :validator/valid? (comp some? last)}

    {:validator/desc "Hard tabs"
     :validator/name :validator.name/hard-tabs
     :validator/target :target/each-line
     :validator/valid? #(not (str/includes? % "\t"))}})

(defn file?
  [x]
  (instance? File x))

(defn scanner-seq
  [^Reader io ^String delim]
  (let [scanner (doto (java.util.Scanner. io)
                  (.useDelimiter delim))]
    (iterator-seq scanner)))

(defn- lint-file
  [^File file]
  (with-open [rdr (io/reader file)]
    (let [lines (scanner-seq rdr "\n")]
      (reduce
       (fn run-validator
         [acc {:keys [validator/desc
                      validator/target
                      validator/valid?]
               :as validator}]
         (case target
           :target/all-lines
           (if (valid? lines)
             acc
             (assoc-in acc [:file/errors validator] true))
           :target/each-line
           (let [[_ errors] (reduce
                             (fn validate-line
                               [[^long line-no errors] line]
                               [(inc line-no)
                                (cond-> errors
                                  (not (valid? line)) (conj line-no))])
                             [1 #{}]
                             lines)]
             (cond-> acc
               (seq errors)
               (assoc-in [:file/errors validator] errors)))))
       {:file/errors {}
        :file/file file
        :file/name (.getName file)}
       validators))))

(defn lint
  ([]
   (comp (map lint-file)
         (map #(assoc % :file/ok? (-> % :file/errors empty?)))))
  ([files]
   {:pre [(or (nil? files) (seq files))
          (every? file? files)]}
   (into #{} (lint) files)))
