(ns flow.cursors)

;; Most of this adapted from either from CLJS core or Om, thanks David
;; Nolan & contributors for the design ideas, inspiration and code :)

;; The Clojure implementation is a small subset of the CLJS cursor
;; functionality (due to the difference in interfaces between Clojure
;; 1.6 and CLJS). It's really only meant for testing Flow.

(defprotocol Cursor
  (-value [_])
  (-!state [_])
  (-path [_])
  (->atom [_ extra-path]))

(defprotocol Keyable
  (-keyed-by [_ f]))

(defn cursor? [v]
  (satisfies? Cursor v))

(defn get-at-path [m [p & more-path :as path]]
  (if (and m (seq path))
    (cond
      (and (vector? p) (= (first p) ::pk))
      (let [[_ key-value] p
            {key-fn ::key-fn} (meta p)]
        (get-at-path (first (filter (comp #(= key-value %) key-fn) m)) more-path))

      (or       (instance? clojure.lang.ILookup m)
                                              
          (nil? m))
      
      (get-at-path (get m p) more-path)

      (number? p)
      (get-at-path (nth m p) more-path))
    
    m))

(defn assoc-at-path [m [p & more-path :as path] v]
  (if (seq path)
    (cond
      (and (vector? p) (= (first p) ::pk))
      (let [[_ key-value] p
            {key-fn ::key-fn} (meta p)]
        (into (empty m) (for [el m]
                          (if (= key-value (key-fn el))
                            (assoc-at-path el more-path v)
                            el))))

      (or       (instance? clojure.lang.Associative m)
                                                   
          (nil? m))
      (assoc m p (assoc-at-path (get m p) more-path v))

      (and (seq? m)
           (number? p))
      (if (zero? p)
        (cons (assoc-at-path (first m) more-path v)
              (rest m))
        (cons (first m)
              (assoc-at-path (rest m) (cons (dec p) more-path) v))))
    
    v))

     
(defn cursor->atom [!state path]
  (reify
    Cursor
    (-value [this] (deref this))
    (-!state [_] !state)
    (-path [_] path)
    (->atom [_ extra-path]
      (cursor->atom !state (vec (concat path extra-path))))
    
    clojure.lang.IDeref
    (deref [this]
      (get-at-path @!state path))))

      
                                
                                      
        
          
                         
                        
                    
                          
                                                           
    
          
                        
                          
                                              
                                  

         
              
                     
                   
    
          
               
                                 

          
                             
                                    
                                                   
                   

         
                    
                                      
                       
                                         
                          
                                            
                             
                                                     

                    
                                  
                                
                                    
                          

         
                 
                           

      
                       
                
                        
                 
                
                                                      

(declare ->cursor)

     
(defn map-cursor [value !state path]
  (reify
    Cursor
    (-value [_] value)
    (-!state [_] !state)
    (-path [_] path)
    (->atom [_ extra-path]
      (cursor->atom !state (vec (concat path extra-path))))
    
    clojure.lang.ILookup
    (valAt [this k]
      (get this k nil))
    (valAt [this k not-found]
      (let [v (get value k not-found)]
        (if-not (= v not-found)
          (->cursor v !state (conj path k))
          not-found)))
    
    clojure.lang.IFn
    (invoke [this k]
      (get this k))
    (invoke [this k not-found]
      (get this k not-found))

    clojure.lang.Seqable
    (seq [this]
      (when (pos? (count value))
        (map (fn [[k v]]
               (clojure.lang.MapEntry. k (->cursor v !state (conj path k))))
             value)))
    
    clojure.lang.IPersistentMap
    (assoc [_ k v]
      (map-cursor (assoc value k v) !state path))
    (without [_ k]
      (map-cursor (dissoc value k) !state path))))

      
                                    
        
          
                      
                        
                    
                          
                                                           
    
             
                            
                                                          
         
              
                   

              
               
                                     
    
            
               
                    

               
                   
                                             

           
                     
                           
                               
                                          
                               
                                           
                      
    
       
                     
                       
                               
                                 

            
                
                                
                        
                                                     
                     
    
                
                         
                               
                   
                                                  

        
                  
                                                 

          
                     
                         
                                
                         

                    
                               
                                       

     
(defn vec-cursor [value !state path key-fn]
  (letfn [(pk [v i]
            (if key-fn
              (with-meta [::pk (key-fn v)]
                {::key-fn key-fn})
              i))]
    (reify
      Cursor
      (-value [_] value)
      (-!state [_] !state)
      (-path [_] path)
      (->atom [_ extra-path]
        (cursor->atom !state (vec (concat path extra-path))))

      Keyable
      (-keyed-by [_ f]
        (vec-cursor value !state path f))
    
      clojure.lang.Sequential

      clojure.lang.IPersistentCollection
      (count [_]
        (count value))

      clojure.lang.ILookup
      (valAt [this n]
        (nth this n nil))
      (valAt [this n not-found]
        (nth this n not-found))

      clojure.lang.IFn
      (invoke [this k]
        (get this k))
      (invoke [this k not-found]
        (get this k not-found))

      clojure.lang.Indexed
      (nth [this n]
        (->cursor (nth value n) !state (conj path n)))
      (nth [this n not-found]
        (if (< n (count value))
          (->cursor (nth value n) !state (conj path n))
          not-found))

      clojure.lang.Seqable
      (seq [this]
        (when (not-empty value)
          (map-indexed (fn [i v]
                         (->cursor v !state (conj path (pk v i))))
                       value)))

      clojure.lang.Associative
      (assoc [this n v]
        (->cursor (assoc value n v) !state path)))))

      
                                           
                   
                      
                                          
                                  
                  
          
            
                        
                          
                      
                            
                                                             

             
                      
                                         
      
                 

               
                              
                                                                   
           
                
                     

                
                 
                                              

              
                 
                      
                 
                  
                                                       

             
                       
                          
                                 
                                

         
                       
                         
                                 
                                   

              
                    
                              
                                                    
      
                              
                                
                                
                                                     
          
                     

              
                  
                               
                                
                                                                  
                               

                  
                           
                                 
                        
                                                             

            
                   
                                                      
                  
                                                     
      
            
                       
                           
                                  
                           
      
                      
                                 
                                          

     
(defn set-cursor [value !state path key-fn]
  (letfn [(pk [v]
            (if key-fn
              (with-meta [::pk (key-fn v)]
                {::key-fn key-fn})
              v))]
    (reify
      Cursor
      (-value [_] value)
      (-!state [_] !state)
      (-path [_] path)
      (->atom [_ extra-path]
        (cursor->atom !state (vec (concat path extra-path))))

      Keyable
      (-keyed-by [_ f]
        (vec-cursor value !state path f))
    
      clojure.lang.Sequential

      clojure.lang.IPersistentCollection
      (count [_]
        (count value))

      clojure.lang.IFn
      (invoke [this k]
        (get this k))
      (invoke [this k not-found]
        (get this k not-found))

      clojure.lang.Seqable
      (seq [this]
        (when (not-empty value)
          (map (fn [v]
                 (->cursor v !state (conj path (pk v))))
               value))))))

      
                                           
                 
                      
                                          
                                  
                  
          
            
                        
                          
                      
                            
                                                             

             
                      
                                         
      
               
                              
                                                                   
      
           
                
                     

                
                 
                                              

              
                 
                      
                 
                  
                                                       

                          
                                          

             
                       
                             
      
                                 
                             
                                                
                     

          
                        
                                                           
      
         
                       
                         
                                 
                                   

              
                  
                               
                      
                                                        
                       

            
                       
                           
                                  
                           
      
                      
                                 
                                          

      
                                           
                
          
                      
                        
                    
                          
                                                           
    
          
                     
                         
                              
                         

(defn ->cursor [value !state path]
  (cond
    (cursor? value) value

    (map? value) (map-cursor value !state path)
    (set? value) (set-cursor value !state path nil)
    (or (coll? value) (seq? value)) (vec-cursor value !state path nil)

                                        
                                                

    :else value))

(defn keyed-by [f coll]
  (cond
    (satisfies? Keyable coll) (-keyed-by coll f)

    (or (sequential? coll)
                              )
    (into (empty coll)
          (map (fn [el]
                 (if (cursor? el)
                   (->cursor (-value el)
                             (-!state el)
                             (concat (butlast (-path el)) [(with-meta [::pk (f el)]
                                                             {::key-fn f})]))
                   el))
               coll))
    
    :else coll))


;;;;;;;;;;;; This file autogenerated from src/flow/cursors.cljx
