/*
 * Decompiled with CFR 0.152.
 */
package com.davidsoergel.dsutils;

import com.davidsoergel.dsutils.StreamUtils;
import com.davidsoergel.dsutils.UrlContentCache;
import com.davidsoergel.dsutils.UrlContentCacheException;
import com.davidsoergel.dsutils.file.FileUtils;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.NoSuchElementException;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.zip.CRC32;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

public class UrlContentCacheImpl
implements UrlContentCache {
    private static final Logger logger = Logger.getLogger(UrlContentCacheImpl.class);
    private static final int bufferSize = 16384;
    private File cacheRootDir;
    private String cacheRootDirCanonicalPath;
    private long maxSize = Long.MAX_VALUE;
    private long cacheSize = 0L;
    private SortedMap<String, File> filesByAccessDate = null;

    public UrlContentCacheImpl(File cacheRootDir) throws UrlContentCacheException {
        try {
            this.cacheRootDirCanonicalPath = cacheRootDir.getCanonicalPath();
        }
        catch (IOException e) {
            logger.error("Error", e);
            throw new UrlContentCacheException(e);
        }
        this.cacheRootDir = new File(this.cacheRootDirCanonicalPath);
        this.init();
    }

    private void init() throws UrlContentCacheException {
        if (!this.cacheRootDir.exists() && !this.cacheRootDir.mkdirs()) {
            throw new UrlContentCacheException("Can't create cache directory: " + this.cacheRootDirCanonicalPath);
        }
        this.filesByAccessDate = new TreeMap<String, File>();
        try {
            this.registerLocalFile(this.cacheRootDir);
        }
        catch (IOException e) {
            logger.error("Error", e);
            throw new UrlContentCacheException(e);
        }
    }

    private void registerLocalFile(File file) throws IOException {
        File[] files;
        if (file.isFile()) {
            this.filesByAccessDate.put("" + this.getLastAccessTime(file) + ":" + file.getCanonicalPath(), file);
            this.cacheSize += file.length();
        }
        if ((files = file.listFiles()) != null) {
            for (File f : files) {
                this.registerLocalFile(f);
            }
        }
    }

    private long getLastAccessTime(File file) throws IOException {
        return this.getChecksumFile(file).lastModified();
    }

    public UrlContentCacheImpl(String cacheRootDir) throws UrlContentCacheException {
        this(new File(cacheRootDir));
    }

    public void setMaxSize(long maxSize) {
        this.maxSize = maxSize;
    }

    public void clear() {
        FileUtils.deleteDirectory(this.cacheRootDir);
    }

    public String getChecksum(String s) throws MalformedURLException, UrlContentCacheException {
        return this.getChecksum(new URL(s));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String getChecksum(URL s) throws UrlContentCacheException {
        String string;
        block7: {
            BufferedReader br = null;
            try {
                br = new BufferedReader(new FileReader(this.getChecksumFile(this.getFile(s))));
                string = br.readLine();
                if (br == null) break block7;
            }
            catch (Throwable throwable) {
                try {
                    if (br != null) {
                        br.close();
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    return this.recalculateChecksum(s);
                }
                catch (UrlContentCacheException e) {
                    return this.recalculateChecksum(s);
                }
            }
            br.close();
        }
        return string;
    }

    public File getFile(String url) throws UrlContentCacheException, MalformedURLException {
        return this.getFile(new URL(url));
    }

    public File getFile(URL url) throws UrlContentCacheException {
        String filename = this.urlToLocalFilename(url);
        File f = new File(filename);
        if (!f.exists()) {
            this.updateFile(url, false);
            f = new File(filename);
        }
        try {
            this.updateLastAccessTime(f);
        }
        catch (IOException e) {
            logger.error("Error", e);
            throw new UrlContentCacheException(e);
        }
        return f;
    }

    public File getFile(URL url, String checksum) throws UrlContentCacheException {
        try {
            File f = this.getFile(url);
            if (!this.verifyChecksum(f, checksum)) {
                this.updateFile(url, true);
            }
            this.updateLastAccessTime(f);
            return f;
        }
        catch (IOException e) {
            logger.error("Error", e);
            throw new UrlContentCacheException(e);
        }
    }

    public File getFile(String url, String checksum) throws UrlContentCacheException, MalformedURLException {
        return this.getFile(new URL(url), checksum);
    }

    public String recalculateChecksum(String s) throws MalformedURLException, UrlContentCacheException {
        return this.recalculateChecksum(new URL(s));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String recalculateChecksum(URL s) throws UrlContentCacheException {
        String string;
        block6: {
            this.updateChecksum(this.getFile(s));
            BufferedReader br = null;
            try {
                br = new BufferedReader(new FileReader(this.getChecksumFile(this.getFile(s))));
                string = br.readLine();
                if (br == null) break block6;
            }
            catch (Throwable throwable) {
                try {
                    if (br != null) {
                        br.close();
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new UrlContentCacheException(e);
                }
            }
            br.close();
        }
        return string;
    }

    private void updateFile(URL url, boolean force) throws UrlContentCacheException {
        try {
            HttpURLConnection conn = (HttpURLConnection)url.openConnection();
            File localFile = new File(this.urlToLocalFilename(url));
            if (localFile.exists() && !force) {
                conn.setIfModifiedSince(localFile.lastModified());
            }
            conn.connect();
            switch (conn.getResponseCode()) {
                case 304: {
                    return;
                }
                case 200: {
                    if (!localFile.delete()) {
                        // empty if block
                    }
                    this.ensureFreeSpace(conn.getContentLength());
                    if (!localFile.getParentFile().mkdirs()) {
                        // empty if block
                    }
                    if (!localFile.createNewFile()) {
                        throw new UrlContentCacheException("Couldn't create " + localFile);
                    }
                    FileOutputStream os = new FileOutputStream(localFile);
                    StreamUtils.pipe(conn.getInputStream(), os);
                    os.close();
                    this.updateChecksum(localFile);
                    break;
                }
                default: {
                    throw new IOException("Can't handle http response: " + conn.getResponseCode() + " " + conn.getResponseMessage());
                }
            }
        }
        catch (IOException e) {
            logger.error("Error", e);
            throw new UrlContentCacheException(e);
        }
    }

    private String urlToLocalFilename(URL url) {
        StringBuffer sb = new StringBuffer(this.cacheRootDirCanonicalPath);
        String path = url.getFile();
        path = StringUtils.join((Object[])path.split("/"), File.separator);
        sb.append(File.separator).append(url.getHost()).append(path);
        return sb.toString();
    }

    private void ensureFreeSpace(int contentLength) throws UrlContentCacheException {
        try {
            while (this.getAvailableSpace() < (long)contentLength) {
                String key = this.filesByAccessDate.firstKey();
                File f = (File)this.filesByAccessDate.get(key);
                this.cacheSize -= f.length();
                if (!f.delete()) {
                    throw new UrlContentCacheException("Couldn't delete: " + f);
                }
                if (!this.getChecksumFile(f).delete()) {
                    throw new UrlContentCacheException("Couldn't delete: " + this.getChecksumFile(f));
                }
                this.filesByAccessDate.remove(key);
            }
        }
        catch (NoSuchElementException e) {
            logger.error("Error", e);
            throw new UrlContentCacheException("Disk too full: cache is empty, and still we can't store " + contentLength + " bytes.");
        }
        catch (IOException e) {
            logger.error("Error", e);
            throw new UrlContentCacheException(e);
        }
    }

    private long getAvailableSpace() {
        return this.maxSize - this.cacheSize;
    }

    private void updateChecksum(File file) throws IOException {
        File c = this.getChecksumFile(file);
        if (!c.delete()) {
            // empty if block
        }
        if (!c.createNewFile()) {
            throw new IOException("Couldn't create " + c);
        }
        FileWriter w = new FileWriter(c);
        w.write(UrlContentCacheImpl.getChecksumString(file));
        w.close();
    }

    private File getChecksumFile(File f) throws IOException {
        return new File(f.getCanonicalPath() + ".crc32");
    }

    private static String getChecksumString(File file) throws IOException {
        int bytesRead;
        long millis = System.currentTimeMillis();
        FileInputStream in = new FileInputStream(file);
        CRC32 checksum = new CRC32();
        checksum.reset();
        byte[] buffer = new byte[16384];
        while ((bytesRead = ((InputStream)in).read(buffer)) >= 0) {
            checksum.update(buffer, 0, bytesRead);
        }
        ((InputStream)in).close();
        millis = System.currentTimeMillis() - millis;
        logger.trace("Calculated checksum for " + file.getName() + " in " + millis / 1000L + " seconds.");
        return "" + checksum.getValue();
    }

    private void updateLastAccessTime(File f) throws IOException {
        if (!this.getChecksumFile(f).setLastModified(System.currentTimeMillis())) {
            throw new IOException("Failed to update access time: " + f);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean verifyChecksum(File file, String checksum) throws IOException {
        BufferedReader br = null;
        try {
            br = new BufferedReader(new FileReader(this.getChecksumFile(file)));
            String s = br.readLine();
            if (s == null) {
                throw new IOException("Couldn't read from " + file);
            }
            String localChecksum = s.trim();
            boolean bl = localChecksum.equals(checksum);
            return bl;
        }
        finally {
            if (br != null) {
                br.close();
            }
        }
    }
}

