(ns swim.utils
  (:require [clojure.string :refer [lower-case capitalize join split trimr trim]]))

(defn distinct-by
  "Returns a lazy sequence of the elements of coll with duplicate keys removed."
  [keyfn coll]
    (let [step (fn step [xs seen]
                   (lazy-seq
                    ((fn [[x :as xs] seen]
                       (when-let [s (seq xs)]
                         (let [k (keyfn x)]
                           (if (contains? seen k)
                             (recur (rest s) seen)
                             (cons x (step (rest s) (conj seen k)))))))
                     xs seen)))]
      (step coll #{})))

      
           
                                                
               
                                            

(defn capitalize-name
  "Capitalize name if it is all lowercase."
  [name]
  (when name
    (if (= name (lower-case name))
      (capitalize name)
      name)))

(defn last-index
  "Returns the index of the last item in vector that matches pred."
  [vector pred]
  (loop [index (dec (count vector))
         items (rseq vector)]
    (when items
      (if (pred (first items))
        index
        (recur (dec index) (next items))))))

(defn first-index
  "Returns the index of the first item in vector that matches pred."
  [vector pred]
  (loop [index 0
         items (seq vector)]
    (when items
      (if (pred (first items))
        index
        (recur (inc index) (next items))))))

(defn replace-last
  "Replace the last item in vector that matches pred with item."
  [vector pred item]
  (if-let [index (last-index vector pred)]
    (assoc vector index item)
    vector))

(defn replace-first
  "Replace the first item in vector that matches pred with item."
  [vector pred item]
  (if-let [index (first-index vector pred)]
    (assoc vector index item)
    vector))

(defn remove-last
  "Remove the last item in vector that matches pred."
  [vector pred]
  (if-let [index (last-index vector pred)]
    (into (subvec vector 0 index) (subvec vector (inc index)))
    vector))

(defn removev [f coll]
  (filterv (complement f) coll))

(defn eq
  "Returns a predicate that returns true when calling key on item returns val. This will commonly
  be used with a keyword for key (eq :name name), though that is not required."
  [key val]
  (fn [item]
    (= (key item) val)))

(defn neq [key val]
  (complement (eq key val)))

(defn update-each
  "Adjust values for the given keys in m by calling f. This is painfully similar to useful's
  update-each, except that it doesn't call the function if the item is missing from the map."
  [m keys f]
  (reduce (fn [m key]
            (cond-> m (contains? m key)
                    (update-in [key] f)))
          m keys))

(defn split-on
  "Split the given collection on the first item for which pred returns true.
  Returns [head-vector tail-seq]"
 [pred coll]
  (loop [head []
         coll (seq coll)]
    (if-let [[x & xs] coll]
      (if (pred x)
        [head coll]
        (recur (conj head x) xs))
      [head])))

(defn index-by
  "Returns a map from the result of calling f on each item in coll to that item."
  [f coll]
  (into {} (for [item coll]
             [(f item) item])))

(defn alternating
  "Returns a lazy seq of the first item in each coll, then the second etc. Unlike interleave, does
  not stop early."
  [& colls]
  (lazy-seq
   (when-let [seqs (seq (keep seq colls))]
     (concat (map first seqs)
             (apply alternating (map rest seqs))))))

(defn user-name [user]
  (when user
    (->> [:first-name :last-name]
         (map user)
         (remove empty?)
         (map trim)
         (join " "))))

(defn user-initials [user]
  (->> [:first-name :last-name]
       (map (comp first user))
       (remove nil?)
       (map capitalize)
       (join)))

(defn truncate-string
  ([string num]
     (truncate-string string num "..."))
  ([string num ellipsis]
     (when string
       (let [string (trim string)
             limit (- num (count ellipsis))]
         (if (< num (count string))
           (str (trimr (subs string 0 limit)) ellipsis)
           string)))))

(defn truncate-string-on-words
  ([string num]
     (truncate-string-on-words string num "..."))
  ([string num ellipsis]
     (when string
       (let [string (trim string)
             limit (- num (count ellipsis))]
         (reduce (fn [result word]
                   (let [next-result (str result " " word)]
                     (if (< limit (count next-result))
                       (reduced (str result ellipsis))
                       next-result)))
                 (split string #"\s+"))))))

(defn truncate-words
  ([string num]
     (truncate-words string num "..."))
  ([string num ellipsis]
     (let [words (->> (split string #"\s+")
                      (take (inc num)))]
       (str (join " " (butlast words))
            (when (< num (count words))
              ellipsis)))))

(defn ? [& args]
  (apply prn args)
  (last args))

;;;;;;;;;;;; This file autogenerated from src/swim/utils.cljx
