(ns swim.parse
  (:refer-clojure :exclude [replace])
  (:require [clojure.string :refer [replace capitalize join split]]
            [swim.utils :refer [split-on eq]]
            [swim.parse.message :refer [tag identifier address email phone sentence]]
            [swim.parse.core :refer [<< >> end many str* trim-whitespace
                                     until-whitespace whitespace word-char]]))

;; Should probably remove parse- from the beginning of all these method names.

(def parse-address
  (comp first (<< (trim-whitespace (address))
                  (end))))

(def parse-email
  (comp first (<< (trim-whitespace (email))
                  (end))))

(def min-phone 10)

(defn phone-digits [string]
  (let [digits (filter #(<= \0 % \9) string)
        num (count digits)]
    (if (or (= \+ (first string))
            (= \1 (first digits)))
      digits
      (when (<= min-phone num)
        (cons \1 digits)))))

(defn parse-phone [string]
  (when-let [digits (phone-digits string)]
    {:type :phone
     :identifier (apply str (cons \+ digits))}))

(defn body-string [body]
  (apply str (map :string body)))

(defn message-anchor [body]
  (first (split (body-string body) #"\n\n")))

(defn parse-subject [body]
  (let [parse (trim-whitespace (sentence :stop-at-newline true))
        string (body-string body)]
    (or (first (parse string))
        string)))

(defn split-last-words
  "Extract the final n words from the provided string."
  [string n]
  (let [parse (many 0 n (>> (whitespace) (many (word-char))))
        [val _ prefix] (parse (reverse string))
        to-str #(apply str (reverse %))]
    [(not-empty (to-str prefix))
     (reverse (filter seq (map to-str val)))]))

;; Keep this until everyone has the new tagging code
(defn keep-trailing-space [first-name]
  (fn [name]
    (cond-> (first-name name)
            (contains? #{\space \tab \newline} (last name))
            (str " "))))

(defn contact-first-name [segment]
  (let [first-name (comp first (str* (trim-whitespace (until-whitespace))))]
    (if (:contact segment)
      (update-in segment [:string] (keep-trailing-space first-name))
      segment)))

(defn sentence-around-user
  "Returns the sentence surrounding the first mention of the specified user. This is used for
  viral invites."
  [body user-id]
  (let [sentence (trim-whitespace (sentence))
        body (map contact-first-name body)
        [head tail] (split-on (eq :user-id user-id) body)]
    (->> [(last (first ((many sentence) (body-string head))))
          (first (sentence (body-string tail)))]
         (remove nil?)
         (join " ")
         (apply str))))
