goog.provide("goog.crypt.Sha2_64bit");
goog.require("goog.array");
goog.require("goog.asserts");
goog.require("goog.crypt.Hash");
goog.require("goog.math.Long");
goog.crypt.Sha2_64bit = function(numHashBlocks, initHashBlocks) {
  goog.crypt.Sha2_64bit.base(this, "constructor");
  this.blockSize = goog.crypt.Sha2_64bit.BLOCK_SIZE_;
  this.chunk_ = goog.global["Uint8Array"] ? new Uint8Array(this.blockSize) : new Array(this.blockSize);
  this.chunkBytes_ = 0;
  this.total_ = 0;
  this.hash_ = [];
  this.numHashBlocks_ = numHashBlocks;
  this.w_ = [];
  this.initHashBlocks_ = goog.crypt.Sha2_64bit.toLongArray_(initHashBlocks);
  this.needsReset_ = false;
  this.reset();
};
goog.inherits(goog.crypt.Sha2_64bit, goog.crypt.Hash);
goog.crypt.Sha2_64bit.BLOCK_SIZE_ = 1024 / 8;
goog.crypt.Sha2_64bit.PADDING_ = [].concat([128], goog.array.repeat(0, goog.crypt.Sha2_64bit.BLOCK_SIZE_ - 1));
goog.crypt.Sha2_64bit.prototype.reset = function() {
  this.chunkBytes_ = 0;
  this.total_ = 0;
  this.hash_ = goog.array.clone(this.initHashBlocks_);
  this.needsReset_ = false;
};
goog.crypt.Sha2_64bit.prototype.update = function(message, opt_length) {
  var length = opt_length !== undefined ? opt_length : message.length;
  if (this.needsReset_) {
    throw new Error("this hasher needs to be reset");
  }
  var chunkBytes = this.chunkBytes_;
  if (typeof message === "string") {
    var i = 0;
    for (; i < length; i++) {
      var b = message.charCodeAt(i);
      if (b > 255) {
        throw new Error("Characters must be in range [0,255]");
      }
      this.chunk_[chunkBytes++] = b;
      if (chunkBytes == this.blockSize) {
        this.computeChunk_();
        chunkBytes = 0;
      }
    }
  } else if (goog.isArrayLike(message)) {
    i = 0;
    for (; i < length; i++) {
      b = message[i];
      if (typeof b !== "number" || b < 0 || b > 255 || b != (b | 0)) {
        throw new Error("message must be a byte array");
      }
      this.chunk_[chunkBytes++] = b;
      if (chunkBytes == this.blockSize) {
        this.computeChunk_();
        chunkBytes = 0;
      }
    }
  } else {
    throw new Error("message must be string or array");
  }
  this.chunkBytes_ = chunkBytes;
  this.total_ += length;
};
goog.crypt.Sha2_64bit.prototype.digest = function() {
  if (this.needsReset_) {
    throw new Error("this hasher needs to be reset");
  }
  var totalBits = this.total_ * 8;
  if (this.chunkBytes_ < 112) {
    this.update(goog.crypt.Sha2_64bit.PADDING_, 112 - this.chunkBytes_);
  } else {
    this.update(goog.crypt.Sha2_64bit.PADDING_, this.blockSize - this.chunkBytes_ + 112);
  }
  var i = 127;
  for (; i >= 112; i--) {
    this.chunk_[i] = totalBits & 255;
    totalBits = totalBits / 256;
  }
  this.computeChunk_();
  var n = 0;
  var digest = new Array(8 * this.numHashBlocks_);
  i = 0;
  for (; i < this.numHashBlocks_; i++) {
    var block = this.hash_[i];
    var high = block.getHighBits();
    var low = block.getLowBits();
    var j = 24;
    for (; j >= 0; j = j - 8) {
      digest[n++] = high >> j & 255;
    }
    j = 24;
    for (; j >= 0; j = j - 8) {
      digest[n++] = low >> j & 255;
    }
  }
  this.needsReset_ = true;
  return digest;
};
goog.crypt.Sha2_64bit.prototype.computeChunk_ = function() {
  var chunk = this.chunk_;
  var K_ = goog.crypt.Sha2_64bit.K_;
  var w = this.w_;
  var i = 0;
  for (; i < 16; i++) {
    var offset = i * 8;
    w[i] = new goog.math.Long(chunk[offset + 4] << 24 | chunk[offset + 5] << 16 | chunk[offset + 6] << 8 | chunk[offset + 7], chunk[offset] << 24 | chunk[offset + 1] << 16 | chunk[offset + 2] << 8 | chunk[offset + 3]);
  }
  i = 16;
  for (; i < 80; i++) {
    var s0 = this.sigma0_(w[i - 15]);
    var s1 = this.sigma1_(w[i - 2]);
    w[i] = this.sum_(w[i - 16], w[i - 7], s0, s1);
  }
  var a = this.hash_[0];
  var b = this.hash_[1];
  var c = this.hash_[2];
  var d = this.hash_[3];
  var e = this.hash_[4];
  var f = this.hash_[5];
  var g = this.hash_[6];
  var h = this.hash_[7];
  i = 0;
  for (; i < 80; i++) {
    var S0 = this.Sigma0_(a);
    var maj = this.majority_(a, b, c);
    var t2 = S0.add(maj);
    var S1 = this.Sigma1_(e);
    var ch = this.choose_(e, f, g);
    var t1 = this.sum_(h, S1, ch, K_[i], w[i]);
    h = g;
    g = f;
    f = e;
    e = d.add(t1);
    d = c;
    c = b;
    b = a;
    a = t1.add(t2);
  }
  this.hash_[0] = this.hash_[0].add(a);
  this.hash_[1] = this.hash_[1].add(b);
  this.hash_[2] = this.hash_[2].add(c);
  this.hash_[3] = this.hash_[3].add(d);
  this.hash_[4] = this.hash_[4].add(e);
  this.hash_[5] = this.hash_[5].add(f);
  this.hash_[6] = this.hash_[6].add(g);
  this.hash_[7] = this.hash_[7].add(h);
};
goog.crypt.Sha2_64bit.prototype.sigma0_ = function(value) {
  var valueLow = value.getLowBits();
  var valueHigh = value.getHighBits();
  var low = valueLow >>> 1 ^ valueHigh << 31 ^ valueLow >>> 8 ^ valueHigh << 24 ^ valueLow >>> 7 ^ valueHigh << 25;
  var high = valueHigh >>> 1 ^ valueLow << 31 ^ valueHigh >>> 8 ^ valueLow << 24 ^ valueHigh >>> 7;
  return new goog.math.Long(low, high);
};
goog.crypt.Sha2_64bit.prototype.sigma1_ = function(value) {
  var valueLow = value.getLowBits();
  var valueHigh = value.getHighBits();
  var low = valueLow >>> 19 ^ valueHigh << 13 ^ valueHigh >>> 29 ^ valueLow << 3 ^ valueLow >>> 6 ^ valueHigh << 26;
  var high = valueHigh >>> 19 ^ valueLow << 13 ^ valueLow >>> 29 ^ valueHigh << 3 ^ valueHigh >>> 6;
  return new goog.math.Long(low, high);
};
goog.crypt.Sha2_64bit.prototype.Sigma0_ = function(value) {
  var valueLow = value.getLowBits();
  var valueHigh = value.getHighBits();
  var low = valueLow >>> 28 ^ valueHigh << 4 ^ valueHigh >>> 2 ^ valueLow << 30 ^ valueHigh >>> 7 ^ valueLow << 25;
  var high = valueHigh >>> 28 ^ valueLow << 4 ^ valueLow >>> 2 ^ valueHigh << 30 ^ valueLow >>> 7 ^ valueHigh << 25;
  return new goog.math.Long(low, high);
};
goog.crypt.Sha2_64bit.prototype.Sigma1_ = function(value) {
  var valueLow = value.getLowBits();
  var valueHigh = value.getHighBits();
  var low = valueLow >>> 14 ^ valueHigh << 18 ^ valueLow >>> 18 ^ valueHigh << 14 ^ valueHigh >>> 9 ^ valueLow << 23;
  var high = valueHigh >>> 14 ^ valueLow << 18 ^ valueHigh >>> 18 ^ valueLow << 14 ^ valueLow >>> 9 ^ valueHigh << 23;
  return new goog.math.Long(low, high);
};
goog.crypt.Sha2_64bit.prototype.choose_ = function(value, one, two) {
  var valueLow = value.getLowBits();
  var valueHigh = value.getHighBits();
  return new goog.math.Long(valueLow & one.getLowBits() | ~valueLow & two.getLowBits(), valueHigh & one.getHighBits() | ~valueHigh & two.getHighBits());
};
goog.crypt.Sha2_64bit.prototype.majority_ = function(one, two, three) {
  return new goog.math.Long(one.getLowBits() & two.getLowBits() | two.getLowBits() & three.getLowBits() | one.getLowBits() & three.getLowBits(), one.getHighBits() & two.getHighBits() | two.getHighBits() & three.getHighBits() | one.getHighBits() & three.getHighBits());
};
goog.crypt.Sha2_64bit.prototype.sum_ = function(one, two, var_args) {
  var low = (one.getLowBits() ^ 2147483648) + (two.getLowBits() ^ 2147483648);
  var high = one.getHighBits() + two.getHighBits();
  var i = arguments.length - 1;
  for (; i >= 2; --i) {
    low = low + (arguments[i].getLowBits() ^ 2147483648);
    high = high + arguments[i].getHighBits();
  }
  if (arguments.length & 1) {
    low = low + 2147483648;
  }
  high = high + (arguments.length >> 1);
  high = high + Math.floor(low / 4294967296);
  return new goog.math.Long(low, high);
};
goog.crypt.Sha2_64bit.toLongArray_ = function(values) {
  goog.asserts.assert(values.length % 2 == 0);
  var result = [];
  var i = 0;
  for (; i < values.length; i = i + 2) {
    result.push(new goog.math.Long(values[i + 1], values[i]));
  }
  return result;
};
goog.crypt.Sha2_64bit.K_ = goog.crypt.Sha2_64bit.toLongArray_([1116352408, 3609767458, 1899447441, 602891725, 3049323471, 3964484399, 3921009573, 2173295548, 961987163, 4081628472, 1508970993, 3053834265, 2453635748, 2937671579, 2870763221, 3664609560, 3624381080, 2734883394, 310598401, 1164996542, 607225278, 1323610764, 1426881987, 3590304994, 1925078388, 4068182383, 2162078206, 991336113, 2614888103, 633803317, 3248222580, 3479774868, 3835390401, 2666613458, 4022224774, 944711139, 264347078, 2341262773, 
604807628, 2007800933, 770255983, 1495990901, 1249150122, 1856431235, 1555081692, 3175218132, 1996064986, 2198950837, 2554220882, 3999719339, 2821834349, 766784016, 2952996808, 2566594879, 3210313671, 3203337956, 3336571891, 1034457026, 3584528711, 2466948901, 113926993, 3758326383, 338241895, 168717936, 666307205, 1188179964, 773529912, 1546045734, 1294757372, 1522805485, 1396182291, 2643833823, 1695183700, 2343527390, 1986661051, 1014477480, 2177026350, 1206759142, 2456956037, 344077627, 2730485921, 
1290863460, 2820302411, 3158454273, 3259730800, 3505952657, 3345764771, 106217008, 3516065817, 3606008344, 3600352804, 1432725776, 4094571909, 1467031594, 275423344, 851169720, 430227734, 3100823752, 506948616, 1363258195, 659060556, 3750685593, 883997877, 3785050280, 958139571, 3318307427, 1322822218, 3812723403, 1537002063, 2003034995, 1747873779, 3602036899, 1955562222, 1575990012, 2024104815, 1125592928, 2227730452, 2716904306, 2361852424, 442776044, 2428436474, 593698344, 2756734187, 3733110249, 
3204031479, 2999351573, 3329325298, 3815920427, 3391569614, 3928383900, 3515267271, 566280711, 3940187606, 3454069534, 4118630271, 4000239992, 116418474, 1914138554, 174292421, 2731055270, 289380356, 3203993006, 460393269, 320620315, 685471733, 587496836, 852142971, 1086792851, 1017036298, 365543100, 1126000580, 2618297676, 1288033470, 3409855158, 1501505948, 4234509866, 1607167915, 987167468, 1816402316, 1246189591]);

//# sourceMappingURL=goog.crypt.sha2_64bit.js.map
