(ns sock.cljs
  (:require [cljs.core.async :refer [chan dropping-buffer put! >! <! close! map< filter<]]
            cljs.core.async.impl.protocols
            [cljs.reader :as reader])
  (:require-macros [cljs.core.async.macros :refer [go go-loop]]))

(def normal-close #{1000})

(defn ws
  "Connects to the given adress via a webSocket and
  returns a set of channels to communicate with the socket.

  This collection is of the form
  `{:in dropping-chan :out dropping-chan :log dropping-chan}`.

  Closing the `:out` channel closes the websocket."
  [address & {:keys [in out err wire-spec]
              :or {in  (chan (dropping-buffer 1024))
                   out (chan (dropping-buffer 1024))
                   err (chan (dropping-buffer 1024))
                   wire-spec
                       {:name    "application/edn"
                        :encoder pr-str
                        :decoder reader/read-string}}}]
  (let [close (chan)
        con (chan)
        socket (js/WebSocket. address)]
    (doto socket
      (aset "onopen" (fn [_]
                       (put! con {:in  in
                                  :out out})))
      (aset "onmessage" (fn [event]
                          (let [msg (.-data event)]
                            (put! in ((:decoder wire-spec) msg)))))
      (aset "onerror"   (fn [event]
                          (let [error (.-data event)]
                            (put! err {:event  :error
                                       :status error}))))
      (aset "onclose" (fn [event]
                        (let [code (.-code event)
                              reason (.-reason event)]
                          (when (not (normal-close code))
                            (put! err {:event  :close
                                       :code   code
                                       :reason reason}))
                          (close! close)))))
    (go
      (<! close)
      (close! close)
      (close! in)
      (close! out)
      (close! err)
      (.close socket))
    (go-loop []
             (if-let [msg (<! out)]
               (do (.send socket ((:encoder wire-spec) msg))
                   (recur))
               (close! close)))
    {:err err
     :con con}))