(ns volga.core
  (:require [clojure.core.async :as async]))

(defn- unit->hash [{:keys [inputs] :as unit}]
  (hash (merge (select-keys unit [:name :config])
               {:inputs (set (mapv unit->hash inputs))})))

(defn- fix-entrypoint [graph]
  (let [entrypoint-hash (first
                         (first
                          (filter (fn [[_ {:keys [name]}]]
                                    (= name :request))
                                  graph)))
        entrypoint-unit (get graph entrypoint-hash)
        entrypoint-outputs (get-in graph [entrypoint-hash :outputs])]
    (loop [output (first entrypoint-outputs)
           outputs (rest entrypoint-outputs)
           dict (-> graph
                    (dissoc entrypoint-hash)
                    (assoc :request entrypoint-unit))]
      (if output
        (recur (first outputs) (rest outputs)
               (update-in dict [output :inputs]
                          (fn [inputs]
                            (-> inputs
                                (disj entrypoint-hash)
                                (conj :request)))))
        dict))))

(defn- inject-context [graph]
  (let [context-unit {:name :context
                      :config {}
                      :inputs #{:request}
                      :outputs (set (map first
                                         (filter (fn [[unit-hash _]]
                                                   (not= unit-hash :request))
                                                 graph)))}
        context-outputs (get context-unit :outputs)]
    (loop [output (first context-outputs)
           outputs (rest context-outputs)
           dict (-> graph
                    (assoc :context context-unit))]
      (if output
        (recur (first outputs) (rest outputs)
               (update-in dict [output :inputs] conj :context))
        dict))))

(defn- fix-inputs [unit entrypoint]
  (let [entrypoint-hash (unit->hash entrypoint)]
    (update unit :inputs
            (fn [inputs]
              (or (not-empty
                   (map (fn [input]
                          (if (or (= input :request) (when (map? input)
                                                       (= entrypoint-hash
                                                          (unit->hash input))))
                            entrypoint
                            (fix-inputs input entrypoint)))
                        inputs))
                  [entrypoint])))))

(defn find-entrypoint* [{:keys [inputs name] :as unit}]
  (if (and (empty? inputs)
           (= name :request))
    unit
    (map find-entrypoint* inputs)))

(defn- find-entrypoint [unit]
  (first (filter identity (flatten (find-entrypoint* unit)))))

(defn unit->graph*
  ([unit]
   (inject-context
    (fix-entrypoint
     (unit->graph* unit (or (find-entrypoint unit)
                            {:name :request
                             :outputs #{}
                             :config {}})))))
  ([unit entrypoint]
   (let [entrypoint (or entrypoint
                        {:name :request
                         :outputs #{}
                         :config {}})
         entrypoint-hash (unit->hash entrypoint)]
     (unit->graph* unit
                   entrypoint
                   {entrypoint-hash entrypoint
                    :response {:name :response
                                :inputs #{}
                                :config {}}})))
  ([unit entrypoint graph]
   (unit->graph* (fix-inputs unit entrypoint) entrypoint graph nil))
  ([{:keys [inputs] :as unit} entrypoint graph parent]
   (let [unit-hash (unit->hash unit)
         graph (update-in graph
                          [(or parent :response) :inputs]
                          conj unit-hash)
         graph (if (contains? graph unit-hash)
                 (update-in graph
                            [unit-hash :outputs]
                            conj (or parent :response))
                 (assoc graph
                        unit-hash
                        (assoc unit
                               :inputs #{}
                               :outputs #{(or parent :response)})))]
     (loop [unit (first inputs) units (rest inputs) graph graph]
       (if unit
         (recur (first units) (rest units)
                (unit->graph* unit entrypoint graph unit-hash))
         graph)))))

(defn- inject-control-channels [graph]
  (->> graph
      (map (fn [[unit-hash unit]]
             (let [output-channel (async/chan)]
               [unit-hash (-> unit
                              (assoc :output-channel output-channel)
                              ((fn [unit]
                                 (if (= unit-hash :response)
                                   unit
                                   (assoc unit :output-channel-mult (async/mult output-channel))))))])))
      (into {})))

(defn- unit-result-merger [& results]
  (let [errors (filter (fn [result]
                         (instance? #?(:clj Throwable
                                       :cljs js/Error) (first (vals result))))
                       results)
        context (filter (fn [result]
                          (= :context (first (keys result))))
                        results)]
    (apply merge
           (concat
            (or (not-empty errors)
                results)
            context))))

(defn- inject-flow-channels [graph]
  (->> graph
      (map (fn [[unit-hash {:keys [inputs] :as unit}]]
             [unit-hash
              (assoc unit
                     :input-channel
                     (if (empty? inputs)
                       (async/chan)
                       (async/map unit-result-merger
                                  (map (fn [[_ {:keys [output-channel-mult]}]]
                                         (let [in-ch (async/chan)]
                                           (async/tap output-channel-mult
                                                      in-ch)))
                                       (select-keys graph inputs)))))]))
      (into {})))

(defn resolve-middleware [middleware]
  (if (fn? middleware)
    middleware
    (resolve (symbol middleware))))

(defn- precompile-middleware [middleware]
  (if (vector? middleware)
    #(apply (resolve-middleware (first middleware))
            %
            (rest middleware))
    (resolve-middleware middleware)))

(defn- wrap-middleware [middleware handler]
  ((apply comp identity (map precompile-middleware middleware)) handler))

(defn- inject-unit-functions [graph unit-fn-builder]
  (->> graph
      (map (fn [[unit-hash {:keys [name
                                  input-channel output-channel
                                  unit-fn middleware] :as unit}]]
             (let [unit-fn (wrap-middleware middleware
                                            (or unit-fn (unit-fn-builder unit)))]
               [unit-hash
                (assoc unit :pipeline
                       (async/pipeline-async
                        #?(:clj (.. Runtime getRuntime availableProcessors)
                           :cljs 4)
                        output-channel
                        (fn [val result]
                          (async/go
                            (async/>! result
                                      (if (and (not (#{:response :context} name))
                                               (some #(instance? #?(:clj Throwable
                                                                    :cljs js/Error) %)
                                                     (vals val)))
                                        val
                                        {name (try
                                                (unit-fn val)
                                                (catch Throwable e
                                                  e))}))
                            (async/close! result)))
                        input-channel
                        false))])))
      (into {})))

(defn- preprocess-units [graph unit-preprocessor]
  (->> graph
      (map (fn [[unit-hash unit]]
             [unit-hash (unit-preprocessor unit)]))
      (into {})))

(defn unit->graph
  ([unit] (unit->graph unit identity))
  ([unit unit-preprocessor]
   (-> unit
       unit->graph*
       (preprocess-units unit-preprocessor))))

(defn graph->pipeline
  ([graph]
   (graph->pipeline graph (fn [_] identity)))
  ([graph unit-fn-builder]
   (-> graph
       inject-control-channels
       inject-flow-channels
       (inject-unit-functions unit-fn-builder))))

(comment

  (def unit
    {:name :some-unit
     :middleware [[volga.middleware.core/wrap-debug clojure.pprint/pprint]]
     :inputs [:request
              {:name :increment-foo
               :middleware [[volga.middleware.core/wrap-debug clojure.pprint/pprint]]
               :inputs [:request]
               :unit-fn (fn [{:keys [request]}]
                          (update-in request [:foo] inc))}]
     :unit-fn (fn [{:keys [request increment-foo]}]
                [request increment-foo])})

  (let [request {:foo 1}
        pipeline (-> unit
                     unit->graph
                     graph->pipeline)]
    (let [input (get-in pipeline [:request :input-channel])
          output (get-in pipeline [:response :output-channel])]
      (async/go (async/>! input request))
      (get-in (async/<!! output) [:response :some-unit])))

  (clojure.pprint/pp)

  )
