(ns volga-firebird.unit.secure-store
  (:require [buddy.core.codecs :as codecs]
            [buddy.core.codecs.base64 :as base64]
            [buddy.core.crypto :as crypto]
            [buddy.core.hash :as hash]
            [volga-firebird.unit :as unit]
            [firebird-config.core :refer [config]]))

(def ^:private algorithm :aes128-cbc-hmac-sha256)

(defn encode [obj]
  (cond
    (map? obj)    (->> obj
                      (map (fn [[k v]]
                             [k (encode v)]))
                      (into {}))
    (string? obj) (let [{key :key
                         iv  :iv} (get config :cipher)
                        key      (hash/sha256 key)
                        iv       (base64/decode iv)]
                    (codecs/bytes->str
                     (base64/encode
                      (crypto/encrypt (codecs/to-bytes obj) key iv {:algorithm algorithm}))))
    :otherwise    obj))

(defn decode [obj]
  (cond
    (map? obj)    (->> obj
                      (map (fn [[k v]]
                             [k (decode v)]))
                      (into {}))
    (string? obj) (let [{key :key
                         iv  :iv} (get config :cipher)
                        key      (hash/sha256 key)
                        iv       (base64/decode iv)]
                    (codecs/bytes->str
                     (crypto/decrypt (base64/decode obj) key iv {:algorithm algorithm})))
    :otherwise    obj))

(defmethod unit/unit->unit-fn "SECURE_STORE"
  [{{data-set :data-set} :config}]
  (fn [request]
    (decode data-set)))

(defmethod unit/unit-preprocessor "SECURE_STORE"
  [unit]
  (update-in unit [:config :data-set] encode))
