(ns volga-firebird.unit.request
  (:require [cheshire.core :as cheshire]
            [java-time :as java-time]
            [clojure.set :refer [intersection]]
            [clojure.string :as string]
            [firebird-cache.core :as cache]
            [reitit.impl :as impl]
            [volga-firebird.unit :as unit]))

(defn- path->hierarchy [path]
  (when path
    (->> (impl/segments path)
        (filter #(not (empty? %)))
        (reduce (fn [x y]
                  (let [p (last x)]
                    (conj x (if p
                              (string/join "/" [p y])
                              y))))
                [])
        (map-indexed #(string/join "/" [%1 %2])))))

(defn- request->context [{{request-accept          "accept"
                           request-accept-encoding "accept-encoding"
                           request-host            "host"
                           request-forwarded-host  "x-forwarded-host"
                           request-forwarded-ip    "x-forwarded-ip"
                           request-user-agent      "user-agent"} :headers
                          request-content-type                   :muuntaja/request
                          request-content-length                 :content-length
                          response-content-type                  :muuntaja/response
                          request-path                           :path-info
                          request-ip                             :remote-addr
                          request-method                         :request-method
                          request-scheme                         :scheme
                          request-body                           :body-params
                          request-query-string                   :query-string
                          {user-id        :id
                           api-key-issuer :issuer
                           api-key-roles  :roles}                :user
                          pipeline-project-id                    :pipeline-project-id}]
  (let [request-host (or request-forwarded-host request-host)
        request-at   (java-time/instant)]
    {:user_id                 user-id
     :api_key_issuer          api-key-issuer
     :api_key_roles           api-key-roles
     :request_accept          request-accept
     :request_accept_encoding request-accept-encoding
     :request_at              request-at
     :request_content_type    (string/join "; " (vals request-content-type))
     :request_content_length  (if (< (or request-content-length 0) 0)
                                0
                                request-content-length)
     :request_hierarchy       (path->hierarchy request-path)
     :request_host            request-host
     :request_ip              (or request-forwarded-ip request-ip)
     :request_method          (-> (or request-method :PREVIEW) name string/upper-case)
     :request_path            request-path
     :request_scheme          (name (or request-scheme :not-available))
     :request_url             (str (name (or request-scheme :not-available))
                                   "://"
                                   request-host
                                   request-path)
     :request_url_query       request-query-string
     :request_user_agent      request-user-agent
     :request_body            (cheshire/generate-string request-body)
     :response_content_type   (string/join "; " (vals response-content-type))
     :response_time           (System/currentTimeMillis)
     :parent_id               nil
     :sub_request             false
     :project_id              pipeline-project-id
     :request_id              (str (java.util.UUID/randomUUID))}))

(defn- out-of-limit? [{{user-id    :id
                        user-roles :roles} :user}
                      {:keys [name max span]}]
  (not (or (not (some #{name} user-roles))
           (cache/inc-ratelimit {:name (str user-id ":" name)
                                 :max  max
                                 :span span}))))

(defmethod unit/unit->unit-fn :request
  [{{:keys [roles rate-limits]} :config}]
  (fn [{request-format      :muuntaja/format
       {user-roles :roles} :user
       :keys               [request-method headers parameters
                            remote-addr uri]
       :as                 request}]
    (let [data (merge parameters
                      {:method      request-method
                       :format      request-format
                       :headers     headers
                       :remote-addr remote-addr
                       :uri         uri
                       :context     (request->context request)})]
      (cond
        (and (not-empty roles)
             (empty? (intersection (set roles)
                                   (set user-roles))))
        (throw (ex-info "You aren’t authenticated – either not authenticated at all or authenticated incorrectly – but please reauthenticate and try again."
                        (merge data
                               {:status 401})))

        (some (partial out-of-limit? request)
              rate-limits)
        (throw (ex-info "You have exceeded the maximum number of requests, try again after a while"
                        (merge data
                               {:status 429})))

        :otherwise data))))
