
(ns speculoos.scratch.parka
  (:require [#?(:clj clojure.spec.alpha cljs.spec.alpha) :as s]
            [spec-tools.data-spec :as ds]
            [spec-tools.core :as st]
            [clojure.spec.gen.alpha :as gen]
            [clojure.test.check.generators :as tcg]
            [speculoos.core :as sp :refer [deft] #?@(:cljs [:include-macro true])]))

;; when we convert json to edn, all keys are converted to keywords
;; even those we would keep as strings
;; this spec allow us to coerce those keys to strings back

(def string-coercion-spec
  (s/spec-impl
    'string-coercion-spec
    (fn [x]
      (when (or (string? x) (keyword? x))
        (str x)))
    (fn [] (s/gen string?))
    true
    identity))

;; qTree
;; ----------------------------------------------------

(deft QTree
      {offset integer?
       count integer?
       level integer?
       upperChild (s/nilable ::QTree)
       lowerChild (s/nilable ::QTree)})

;; histograms
;; ----------------------------------------------------

(deft SmallHistogram
      [values :- (s/map-of string-coercion-spec integer?)])

#_(map->SmallHistogram {:values {"aze" 21}})

(deft LargeHistogram
      {negatives (s/nilable ::QTree)
       countZero integer?
       positives (s/nilable ::QTree)})

(s/def ::Histogram
  (s/spec-impl
    'histo-spec
    (fn [x]
      (let [ret (s/conform ::SmallHistogram x)]
        (if-not (= ::s/invalid ret)
          ret (s/conform ::LargeHistogram x))))
    (fn []
      (tcg/frequency
        [[1 (s/gen ::LargeHistogram)]
         [20 (s/gen ::SmallHistogram)]]))
    true
    identity))

;; describe
;; ----------------------------------------------------

(deft
  Describe
  {count integer?
   histograms (s/map-of string-coercion-spec ::Histogram)
   counts (s/map-of string-coercion-spec integer?)})

(deft
  DescribeByRow
  {count integer?
   byColumn (s/map-of string-coercion-spec ::Describe)})

;; delta
;; ----------------------------------------------------

(deft
  Delta
  {nEqual integer?
   nNotEqual integer?
   describe (ds/spec ::DeltaDescribe
              {:left ::Describe
               :right ::Describe})
   error (s/nilable ::Describe)})

(deft
  DeltaByRow
  {count integer?
   byColumn (s/map-of string-coercion-spec ::Delta)})

;; inner outer
;; ----------------------------------------------------

(def inner_countDeltaByRow-entry
  (ds/spec ::Inner.countDeltaByRow.entry
           {:key (s/coll-of string-coercion-spec :kind vector?)
            :value ::DeltaByRow}))

(deft
  Inner
  {countRowEqual integer?
   countRowNotEqual integer?
   equalRows ::DescribeByRow
   countDeltaByRow
   (s/coll-of (ds/spec ::InnerCountDeltaByRow
                       {:key (s/coll-of string-coercion-spec :kind vector?)
                        :value ::DeltaByRow})
              :kind vector?)})

;; TODO both as a parametric spec?
(deft
  Outer
  [both :- (ds/spec
             {:left ::DescribeByRow
              :right ::DescribeByRow})])

;; result and analysis
;; ----------------------------------------------------

(deft
  ParkaResult
  {inner ::Inner
   outer ::Outer})

(deft
  DatasetInfo
  {source (s/coll-of string-coercion-spec :kind vector?)
   nStage integer?})

(deft
  ParkaAnalysis
  {result ::ParkaResult
   datasetInfo
   (ds/spec
     ::ParkaAnalysisDatasetInfo
     {:left ::DatasetInfo
      :right ::DatasetInfo})})

(defn generate-analysis
  "this is slow!"
  []
  (gen/generate (s/gen ::ParkaAnalysis)))

(defn conform-report [rep]
  (s/conform ::ParkaAnalysis rep))


;; scratch
;; --------------------------------------------------------

(comment

  (spit "resources/public/two.edn"
        (with-out-str
          (clojure.pprint/pprint
            (gen/generate (s/gen ::ParkaAnalysis)))))

  (gen/generate (s/gen ::Describe))
  (gen/generate (s/gen ::DescribeByRow))
  (gen/generate (s/gen ::Inner))
  (gen/generate (s/gen ::DeltaByRow))
  (first (gen/sample (s/gen ::QTree) 10))
  (gen/generate (s/gen ::QTree))
  (gen/generate (s/gen ::LargeHistogram))
  (gen/sample (s/gen ::LargeHistogram) 100)

  )

(s/def ::int integer?)

;; shorthand syntax

(deft num {val ::int
           bal integer?
           inner1 (s/nilable ::num)
           inner2 (s/nilable ::num)})

(s/invalid? (s/explain ::ParkaAnalysis (read-string (slurp "resources/public/two.edn"))))

