(ns heyoka.client
  (:require [clj-http.client :as h]
            [jsonista.core :as json]
            [heyoka.utils :refer [kebabize-key
                                  params->multiparts]]
            [clojure.string :as str])
  (:import java.io.File))

(def ^:private api-endpoint
  "API endpoint used for HTTP calls"
  "https://api.telegram.org/bot")


(def object-mapper
  (json/object-mapper {:encode-key-fn name
                       :decode-key-fn (comp keyword kebabize-key)}))

(defn- build-request-url
  "Build url for request, with token"
  {:added "0.1.0"}
  [method ^String token]
  (str api-endpoint token "/" (name method)))


(defn- generate-telegram-error
  [{:keys [description error-code] :as full} payload]
  (merge (condp = error-code
           404 {:telegram/not-found description}

           {:telegram/unknown-error description})
         {:response full :payload payload}))


(defn telegram-request
  "High level API request method"
  {:added "0.1.0"}
  ([token method] (telegram-request token method {}))
  ([token method params] (telegram-request token method params nil))
  ([token method params multipart-data]
   (let [target (build-request-url method token)
         {:keys [body]}
         (h/post target
                 (cond-> {:throw-exceptions false}
                   (seq params)    (assoc :query-params params
                                          :content-type :json)
                   ;; Remove content type for multiparts request, otherwise
                   ;; the POST call will fail.
                   multipart-data  (-> (assoc :multipart multipart-data)
                                       (dissoc :query-params :content-type))))
         {:keys [ok result] :as full} (json/read-value body object-mapper)]
     (if ok result
         (throw (ex-info "Telegram API Call Exception"
                         (generate-telegram-error full params)))))))

(defn telegram-request-upload
  "Multipart upload API request method"
  {:added "0.1.0"}
  [token payload ^File file method]
  (let [multipart-data (concat (params->multiparts payload)
                               [{:part-name (name method) :content file
                                 :name (.getName file)}])]
    (telegram-request token (keyword (str "send" (str/capitalize (name method))))
                        payload multipart-data)))
