(ns heyoka.poller
  (:require [clojure.core.async :as a]
            [heyoka.api :as api]
            [clojure.string :as str]))


(defn- async-updates*
  "Retrieve update calls to `core.async` chan that
   will be computed to handler"
  {:added "0.1.0"}
  [token opts]
  (let [chan (a/chan)]
    (a/go
      (try
        (let [results (api/get-updates token opts)]
          (a/>! chan results)
          (a/close! chan))
        (catch Exception _
          (a/>! chan :telegram/failure)
          (a/close! chan))))
    chan))


(defn- resolve-latest-offset
  "Generate an offset from the latest update id record"
  {:added "0.1.0"}
  [updates fallback]
  (if (seq updates)
    (-> updates last :update-id inc)
    fallback))


(defn- process-from-spec!
  [spec {:keys [message]}]
  (let [{:keys [text]} message
        kw (keyword (str/replace text "/" ""))
        executor (get spec kw)]
    (when (fn? executor)
      (a/thread (executor message)))))


(defn- start-poller*
  ([token spec] (start-poller* token spec {:timeout 5}))
  ([token spec {:keys [timeout]}]
   (let [controller   (a/chan)
         error-chan   (a/chan)
         updates-chan (a/chan)]
     (a/go-loop []
       (when-let [data (a/<! updates-chan)]
         (try
           (process-from-spec! spec data)
           (catch Exception _
             (println "erorr")
             (a/>! error-chan :processing/error)))
         (recur)))
     (a/go-loop [offset 0]
       (let [waiter (a/go (a/<! (a/timeout (* 1000 timeout 10)))
                          :telegram/timeout)
             update-results (async-updates* token {:offset offset})
             [status _] (a/alts! [controller update-results waiter error-chan])]
         (condp = status
           ;; Error Collection
           nil               (do (a/close! waiter) (a/close! updates-chan))
           :telegram/timeout (do (a/close! waiter) (a/close! updates-chan))
           :telegram/failure (do (a/close! controller) (a/close! updates-chan))
           :processing/error (do (a/close! controller) (a/close! updates-chan))

           ;; We have data
           (do
             (a/close! waiter)
             (doseq [u status] (a/>! update-results u))
             (recur (resolve-latest-offset status offset))))))
     controller)))


(defn start-poller!
  ([token spec] (start-poller! token spec {}))
  ([token spec options]
   (start-poller* token spec options)))


(defn stop-poller!
  "S"
  [poller]
  (a/close! poller)
  :closed)
