(ns heyoka.api
  (:require [heyoka.client :as c]
            [schema.core :as s]
            [heyoka.models :refer :all]
            [heyoka.utils :refer [params->multiparts]]
            [clojure.java.io :as io])
  (:import java.io.File))


(defn whoami
  "A simple method for testing your bot's authentication token.
  Requires no parameters.

  Returns basic information about the bot in form of a User object."
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token]
  (c/telegram-request token :getMe))


(defn logout
  "Use this method to log out from the cloud Bot API server before launching the bot locally.

  You must log out the bot before running it locally, otherwise there is no
  guarantee that the bot will receive updates. After a successful call,
  you can immediately log in on a local server, but will not be able to log in
  back to the cloud Bot API server for 10 minutes.

  Returns True on success. Requires no parameters."
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token]
  (c/telegram-request token :logOut))


(defn close
  "Use this method to close the bot instance before moving it from one local
   server to another.

  You need to delete the webhook before calling this method to ensure that the
  bot isn't launched again after server restart.
  The method will return error 429 in the first 10 minutes after the bot is
  launched.

  Returns True on success. Requires no parameters."
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token]
  (c/telegram-request token :close))


(defn get-updates
  "Method used to receive incoming updates using long polling.

  Returns an Array of Update objects.

  |    Parameter    |       Type      | Required |                                                                                                                                                                                                                                                                             Description                                                                                                                                                                                                                                                                            |
  |:---------------:|:---------------:|:--------:|:------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------:|
  | offset          | Integer         | Optional | Identifier  of the first update to be returned. Must be greater by one than the  highest among the identifiers of previously received updates. By  default, updates starting with the earliest unconfirmed update are  returned. An update is considered confirmed as soon as getUpdates is called with an offset higher than its update_id. The negative offset can be specified to retrieve updates starting from -offset update from the end of the updates queue. All previous updates will be forgotten.                                                      |
  | limit           | Integer         | Optional | Limits the number of updates to be retrieved. Values between 1-100 are accepted. Defaults to 100.                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
  | timeout         | Integer         | Optional | Timeout  in seconds for long polling. Defaults to 0, i.e. usual short polling.  Should be positive, short polling should be used for testing purposes  only.                                                                                                                                                                                                                                                                                                                                                                                                       |
  | allowed_updates | Array of String | Optional | A  JSON-serialized list of the update types you want your bot to receive.  For example, specify [“message”, “edited_channel_post”,  “callback_query”] to only receive updates of these types. See Update for a complete list of available update types. Specify an empty list to receive all update types except chat_member (default). If not specified, the previous setting will be used.  Please  note that this parameter doesn't affect updates created before the call  to the getUpdates, so unwanted updates may be received for a short  period of time. |


  Notes
  1. This method will not work if an outgoing webhook is set up.
  2. In order to avoid getting duplicate updates, recalculate offset after each server response."
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token opts]
  (c/telegram-request token :getUpdates opts))


(defn send-message
  "Use this method to send text messages.

  On success, the sent Message is returned.


  Parameters:

  |          Parameter          |                                       Type                                       | Required |                                                                                   Description                                                                                  |
  |:---------------------------:|:--------------------------------------------------------------------------------:|:--------:|:------------------------------------------------------------------------------------------------------------------------------------------------------------------------------:|
  | chat_id                     | Integer or String                                                                | Yes      | Unique identifier for the target chat or username of the target channel (in the format @channelusername)                                                                       |
  | message_thread_id           | Integer                                                                          | Optional | Unique identifier for the target message thread (topic) of the forum; for forum supergroups only                                                                               |
  | text                        | String                                                                           | Yes      | Text of the message to be sent, 1-4096 characters after entities parsing                                                                                                       |
  | parse_mode                  | String                                                                           | Optional | Mode for parsing entities in the message text. See formatting options for more details.                                                                                        |
  | entities                    | Array of MessageEntity                                                           | Optional | A JSON-serialized list of special entities that appear in message text, which can be specified instead of parse_mode                                                           |
  | disable_web_page_preview    | Boolean                                                                          | Optional | Disables link previews for links in this message                                                                                                                               |
  | disable_notification        | Boolean                                                                          | Optional | Sends the message silently. Users will receive a notification with no sound.                                                                                                   |
  | protect_content             | Boolean                                                                          | Optional | Protects the contents of the sent message from forwarding and saving                                                                                                           |
  | reply_to_message_id         | Integer                                                                          | Optional | If the message is a reply, ID of the original message                                                                                                                          |
  | allow_sending_without_reply | Boolean                                                                          | Optional | Pass True if the message should be sent even if the specified replied-to message is not found                                                                                  |
  | reply_markup                | InlineKeyboardMarkup or ReplyKeyboardMarkup or ReplyKeyboardRemove or ForceReply | Optional | Additional interface options. A JSON-serialized object for an inline keyboard, custom reply keyboard, instructions to remove reply keyboard or to force a reply from the user. |

  ** Formatting options

  The Bot API supports basic formatting for messages. You can use bold, italic,
  underlined, strikethrough, and spoiler text, as well as inline links and pre-formatted code in your bots' messages.
  Telegram clients will render them accordingly.
  You can specify text entities directly, or use markdown-style or HTML-style formatting.

  Note that Telegram clients will display an alert to the user before opening an inline link
  ('Open this link?' together with the full URL).

  Message entities can be nested, providing following restrictions are met:
  - If two entities have common characters, then one of them is fully contained inside another.
  - bold, italic, underline, strikethrough, and spoiler entities can contain and can be part of any other entities, except pre and code.
  - All other entities can't contain each other.

  Links tg://user?id=<user_id> can be used to mention a user by their ID without using a username. Please note:

    These links will work only if they are used inside an inline link or in an inline keyboard button. For example, they will not work, when used in a message text.
    Unless the user is a member of the chat where they were mentioned, these mentions are only guaranteed to work if the user has contacted the bot in private in the past or has sent a callback query to the bot via an inline button and doesn't have Forwarded Messages privacy enabled for the bot."
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token payload]
  (s/validate MessagePayload payload)
  (c/telegram-request token :sendMessage
                      payload))


(defn forward-message
  "Use this method to forward messages of any kind.

  Service messages can't be forwarded.

  On success, the sent Message is returned.

  |       Parameter      |        Type       | Required |                                                       Description                                                       |
  |:--------------------:|:-----------------:|:--------:|:-----------------------------------------------------------------------------------------------------------------------:|
  | chat_id              | Integer or String | Yes      | Unique identifier for the target chat or username of the target channel (in the format @channelusername)                |
  | message_thread_id    | Integer           | Optional | Unique identifier for the target message thread (topic) of the forum; for forum supergroups only                        |
  | from_chat_id         | Integer or String | Yes      | Unique identifier for the chat where the original message was sent (or channel username in the format @channelusername) |
  | disable_notification | Boolean           | Optional | Sends the message silently. Users will receive a notification with no sound.                                            |
  | protect_content      | Boolean           | Optional | Protects the contents of the forwarded message from forwarding and saving                                               |
  | message_id           | Integer           | Yes      | Message identifier in the chat specified in from_chat_id                                                                |"
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token payload]
  (s/validate MessageForwardPayload payload)
  (c/telegram-request token :forwardMessage
                      payload))


(defn copy-message
  "Use this method to copy messages of any kind.

  Service messages and invoice messages can't be copied.
  A quiz poll can be copied only if the value of the field correct_option_id is
  known to the bot. The method is analogous to the method forwardMessage, but
  the copied message doesn't have a link to the original message.

  Returns the MessageId of the sent message on success.


  |          Parameter          |                                       Type                                       | Required |                                                                                   Description                                                                                  |
  |:---------------------------:|:--------------------------------------------------------------------------------:|:--------:|:------------------------------------------------------------------------------------------------------------------------------------------------------------------------------:|
  | chat_id                     | Integer or String                                                                | Yes      | Unique identifier for the target chat or username of the target channel (in the format @channelusername)                                                                       |
  | message_thread_id           | Integer                                                                          | Optional | Unique identifier for the target message thread (topic) of the forum; for forum supergroups only                                                                               |
  | from_chat_id                | Integer or String                                                                | Yes      | Unique identifier for the chat where the original message was sent (or channel username in the format @channelusername)                                                        |
  | message_id                  | Integer                                                                          | Yes      | Message identifier in the chat specified in from_chat_id                                                                                                                       |
  | caption                     | String                                                                           | Optional | New caption for media, 0-1024 characters after entities parsing. If not specified, the original caption is kept                                                                |
  | parse_mode                  | String                                                                           | Optional | Mode for parsing entities in the new caption. See formatting options for more details.                                                                                         |
  | caption_entities            | Array of MessageEntity                                                           | Optional | A JSON-serialized list of special entities that appear in the new caption, which can be specified instead of parse_mode                                                        |
  | disable_notification        | Boolean                                                                          | Optional | Sends the message silently. Users will receive a notification with no sound.                                                                                                   |
  | protect_content             | Boolean                                                                          | Optional | Protects the contents of the sent message from forwarding and saving                                                                                                           |
  | reply_to_message_id         | Integer                                                                          | Optional | If the message is a reply, ID of the original message                                                                                                                          |
  | allow_sending_without_reply | Boolean                                                                          | Optional | Pass True if the message should be sent even if the specified replied-to message is not found                                                                                  |
  | reply_markup                | InlineKeyboardMarkup or ReplyKeyboardMarkup or ReplyKeyboardRemove or ForceReply | Optional | Additional interface options. A JSON-serialized object for an inline keyboard, custom reply keyboard, instructions to remove reply keyboard or to force a reply from the user. |"
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token payload]
  (s/validate MessageCopyPayload payload)
  (c/telegram-request token :copyMessage payload))


(defn send-photo
  "Use this method to send photos.

  On success, the sent Message is returned.

  |          Parameter          |                                       Type                                       | Required |                                                                                                                                                                                                             Description                                                                                                                                                                                                            |
  |:---------------------------:|:--------------------------------------------------------------------------------:|:--------:|:----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------:|
  | chat_id                     | Integer or String                                                                | Yes      | Unique identifier for the target chat or username of the target channel (in the format @channelusername)                                                                                                                                                                                                                                                                                                                           |
  | message_thread_id           | Integer                                                                          | Optional | Unique identifier for the target message thread (topic) of the forum; for forum supergroups only                                                                                                                                                                                                                                                                                                                                   |
  | photo                       | InputFile or String                                                              | Yes      | Photo  to send. Pass a file_id as String to send a photo that exists on the  Telegram servers (recommended), pass an HTTP URL as a String for  Telegram to get a photo from the Internet, or upload a new photo using  multipart/form-data. The photo must be at most 10 MB in size. The  photo's width and height must not exceed 10000 in total. Width and  height ratio must be at most 20. More information on Sending Files » |
  | caption                     | String                                                                           | Optional | Photo caption (may also be used when resending photos by file_id), 0-1024 characters after entities parsing                                                                                                                                                                                                                                                                                                                        |
  | parse_mode                  | String                                                                           | Optional | Mode for parsing entities in the photo caption. See formatting options for more details.                                                                                                                                                                                                                                                                                                                                           |
  | caption_entities            | Array of MessageEntity                                                           | Optional | A JSON-serialized list of special entities that appear in the caption, which can be specified instead of parse_mode                                                                                                                                                                                                                                                                                                                |
  | has_spoiler                 | Boolean                                                                          | Optional | Pass True if the photo needs to be covered with a spoiler animation                                                                                                                                                                                                                                                                                                                                                                |
  | disable_notification        | Boolean                                                                          | Optional | Sends the message silently. Users will receive a notification with no sound.                                                                                                                                                                                                                                                                                                                                                       |
  | protect_content             | Boolean                                                                          | Optional | Protects the contents of the sent message from forwarding and saving                                                                                                                                                                                                                                                                                                                                                               |
  | reply_to_message_id         | Integer                                                                          | Optional | If the message is a reply, ID of the original message                                                                                                                                                                                                                                                                                                                                                                              |
  | allow_sending_without_reply | Boolean                                                                          | Optional | Pass True if the message should be sent even if the specified replied-to message is not found                                                                                                                                                                                                                                                                                                                                      |
  | reply_markup                | InlineKeyboardMarkup or ReplyKeyboardMarkup or ReplyKeyboardRemove or ForceReply | Optional | Additional interface options. A JSON-serialized object for an inline keyboard, custom reply keyboard, instructions to remove reply keyboard or to force a reply from the user.                                                                                                                                                                                                                                                     |"
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token payload ^File photo]
  (c/telegram-request-upload token payload photo :photo))


(defn send-audio
  "Use this method to send audio files, if you want Telegram clients to display them in the music player.
  Your audio must be in the .MP3 or .M4A format. On success, the sent Message is returned.

  Bots can currently send audio files of up to 50 MB in size, this limit may be changed in the future.
  For sending voice messages, use the `send-voice` method instead.

  |          Parameter          |                                       Type                                       | Required |                                                                                                                                                                                                                                                                     Description                                                                                                                                                                                                                                                                    |
  |:---------------------------:|:--------------------------------------------------------------------------------:|:--------:|:--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------:|
  | chat_id                     | Integer or String                                                                | Yes      | Unique identifier for the target chat or username of the target channel (in the format @channelusername)                                                                                                                                                                                                                                                                                                                                                                                                                                           |
  | message_thread_id           | Integer                                                                          | Optional | Unique identifier for the target message thread (topic) of the forum; for forum supergroups only                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
  | audio                       | InputFile or String                                                              | Yes      | Audio  file to send. Pass a file_id as String to send an audio file that  exists on the Telegram servers (recommended), pass an HTTP URL as a  String for Telegram to get an audio file from the Internet, or upload a  new one using multipart/form-data. More information on Sending Files »                                                                                                                                                                                                                                                     |
  | caption                     | String                                                                           | Optional | Audio caption, 0-1024 characters after entities parsing                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
  | parse_mode                  | String                                                                           | Optional | Mode for parsing entities in the audio caption. See formatting options for more details.                                                                                                                                                                                                                                                                                                                                                                                                                                                           |
  | caption_entities            | Array of MessageEntity                                                           | Optional | A JSON-serialized list of special entities that appear in the caption, which can be specified instead of parse_mode                                                                                                                                                                                                                                                                                                                                                                                                                                |
  | duration                    | Integer                                                                          | Optional | Duration of the audio in seconds                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
  | performer                   | String                                                                           | Optional | Performer                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |
  | title                       | String                                                                           | Optional | Track name                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
  | thumbnail                   | InputFile or String                                                              | Optional | Thumbnail  of the file sent; can be ignored if thumbnail generation for the file  is supported server-side. The thumbnail should be in JPEG format and  less than 200 kB in size. A thumbnail's width and height should not  exceed 320. Ignored if the file is not uploaded using  multipart/form-data. Thumbnails can't be reused and can be only uploaded  as a new file, so you can pass “attach://<file_attach_name>” if  the thumbnail was uploaded using multipart/form-data under  <file_attach_name>. More information on Sending Files » |
  | disable_notification        | Boolean                                                                          | Optional | Sends the message silently. Users will receive a notification with no sound.                                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
  | protect_content             | Boolean                                                                          | Optional | Protects the contents of the sent message from forwarding and saving                                                                                                                                                                                                                                                                                                                                                                                                                                                                               |
  | reply_to_message_id         | Integer                                                                          | Optional | If the message is a reply, ID of the original message                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
  | allow_sending_without_reply | Boolean                                                                          | Optional | Pass True if the message should be sent even if the specified replied-to message is not found                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
  | reply_markup                | InlineKeyboardMarkup or ReplyKeyboardMarkup or ReplyKeyboardRemove or ForceReply | Optional | Additional interface options. A JSON-serialized object for an inline keyboard, custom reply keyboard, instructions to remove reply keyboard or to force a reply from the user.                                                                                                                                                                                                                                                                                                                                                                     |
  "
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token payload ^File audio]
  (c/telegram-request-upload token payload audio :audio))


(defn send-document
  "Use this method to send general files. On success, the sent Message is returned.

  Bots can currently send files of any type of up to 50 MB in size, this limit may be changed in the future.

  |            Parameter           |                                       Type                                       | Required |                                                                                                                                                                                                                                                                    Description                                                                                                                                                                                                                                                                    |
  |:------------------------------:|:--------------------------------------------------------------------------------:|:--------:|:-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------:|
  | chat_id                        | Integer or String                                                                | Yes      | Unique identifier for the target chat or username of the target channel (in the format @channelusername)                                                                                                                                                                                                                                                                                                                                                                                                                                          |
  | message_thread_id              | Integer                                                                          | Optional | Unique identifier for the target message thread (topic) of the forum; for forum supergroups only                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
  | document                       | InputFile or String                                                              | Yes      | File to send. Pass a file_id as String to send a file that exists on  the Telegram servers (recommended), pass an HTTP URL as a String for  Telegram to get a file from the Internet, or upload a new one using  multipart/form-data. More information on Sending Files »                                                                                                                                                                                                                                                                         |
  | thumbnail                      | InputFile or String                                                              | Optional | Thumbnail of the file sent; can be ignored if thumbnail generation  for the file is supported server-side. The thumbnail should be in JPEG  format and less than 200 kB in size. A thumbnail's width and height  should not exceed 320. Ignored if the file is not uploaded using  multipart/form-data. Thumbnails can't be reused and can be only uploaded  as a new file, so you can pass “attach://<file_attach_name>” if  the thumbnail was uploaded using multipart/form-data under  <file_attach_name>. More information on Sending Files » |
  | caption                        | String                                                                           | Optional | Document caption (may also be used when resending documents by file_id), 0-1024 characters after entities parsing                                                                                                                                                                                                                                                                                                                                                                                                                                 |
  | parse_mode                     | String                                                                           | Optional | Mode for parsing entities in the document caption. See formatting options for more details.                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
  | caption_entities               | Array of MessageEntity                                                           | Optional | A JSON-serialized list of special entities that appear in the caption, which can be specified instead of parse_mode                                                                                                                                                                                                                                                                                                                                                                                                                               |
  | disable_content_type_detection | Boolean                                                                          | Optional | Disables automatic server-side content type detection for files uploaded using multipart/form-data                                                                                                                                                                                                                                                                                                                                                                                                                                                |
  | disable_notification           | Boolean                                                                          | Optional | Sends the message silently. Users will receive a notification with no sound.                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
  | protect_content                | Boolean                                                                          | Optional | Protects the contents of the sent message from forwarding and saving                                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
  | reply_to_message_id            | Integer                                                                          | Optional | If the message is a reply, ID of the original message                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             |
  | allow_sending_without_reply    | Boolean                                                                          | Optional | Pass True if the message should be sent even if the specified replied-to message is not found                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
  | reply_markup                   | InlineKeyboardMarkup or ReplyKeyboardMarkup or ReplyKeyboardRemove or ForceReply | Optional | Additional interface options. A JSON-serialized object for an inline keyboard, custom reply keyboard, instructions to remove reply keyboard or to force a reply from the user.                                                                                                                                                                                                                                                                                                                                                                    |
  "
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token payload ^File document]
  (c/telegram-request-upload token payload document :document))


(defn send-video
  "Use this method to send video files, Telegram clients support MPEG4 videos (other formats may be sent as Document).

  On success, the sent Message is returned. Bots can currently send video files of up to 50 MB in size, this limit may be changed in the future.

  |          Parameter          |                                       Type                                       | Required |                                                                                                                                                                                                                                                                    Description                                                                                                                                                                                                                                                                    |
  |:---------------------------:|:--------------------------------------------------------------------------------:|:--------:|:-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------:|
  | chat_id                     | Integer or String                                                                | Yes      | Unique identifier for the target chat or username of the target channel (in the format @channelusername)                                                                                                                                                                                                                                                                                                                                                                                                                                          |
  | message_thread_id           | Integer                                                                          | Optional | Unique identifier for the target message thread (topic) of the forum; for forum supergroups only                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
  | video                       | InputFile or String                                                              | Yes      | Video to send. Pass a file_id as String to send a video that exists  on the Telegram servers (recommended), pass an HTTP URL as a String for  Telegram to get a video from the Internet, or upload a new video using  multipart/form-data. More information on Sending Files »                                                                                                                                                                                                                                                                    |
  | duration                    | Integer                                                                          | Optional | Duration of sent video in seconds                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
  | width                       | Integer                                                                          | Optional | Video width                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
  | height                      | Integer                                                                          | Optional | Video height                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
  | thumbnail                   | InputFile or String                                                              | Optional | Thumbnail of the file sent; can be ignored if thumbnail generation  for the file is supported server-side. The thumbnail should be in JPEG  format and less than 200 kB in size. A thumbnail's width and height  should not exceed 320. Ignored if the file is not uploaded using  multipart/form-data. Thumbnails can't be reused and can be only uploaded  as a new file, so you can pass “attach://<file_attach_name>” if  the thumbnail was uploaded using multipart/form-data under  <file_attach_name>. More information on Sending Files » |
  | caption                     | String                                                                           | Optional | Video caption (may also be used when resending videos by file_id), 0-1024 characters after entities parsing                                                                                                                                                                                                                                                                                                                                                                                                                                       |
  | parse_mode                  | String                                                                           | Optional | Mode for parsing entities in the video caption. See formatting options for more details.                                                                                                                                                                                                                                                                                                                                                                                                                                                          |
  | caption_entities            | Array of MessageEntity                                                           | Optional | A JSON-serialized list of special entities that appear in the caption, which can be specified instead of parse_mode                                                                                                                                                                                                                                                                                                                                                                                                                               |
  | has_spoiler                 | Boolean                                                                          | Optional | Pass True if the video needs to be covered with a spoiler animation                                                                                                                                                                                                                                                                                                                                                                                                                                                                               |
  | supports_streaming          | Boolean                                                                          | Optional | Pass True if the uploaded video is suitable for streaming                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
  | disable_notification        | Boolean                                                                          | Optional | Sends the message silently. Users will receive a notification with no sound.                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
  | protect_content             | Boolean                                                                          | Optional | Protects the contents of the sent message from forwarding and saving                                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
  | reply_to_message_id         | Integer                                                                          | Optional | If the message is a reply, ID of the original message                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             |
  | allow_sending_without_reply | Boolean                                                                          | Optional | Pass True if the message should be sent even if the specified replied-to message is not found                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
  | reply_markup                | InlineKeyboardMarkup or ReplyKeyboardMarkup or ReplyKeyboardRemove or ForceReply | Optional | Additional interface options. A JSON-serialized object for an inline keyboard, custom reply keyboard, instructions to remove reply keyboard or to force a reply from the user.                                                                                                                                                                                                                                                                                                                                                                    |
  "
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token payload ^File video]
  (c/telegram-request-upload token payload video :video))


(defn send-animation
  "Use this method to send animation files (GIF or H.264/MPEG-4 AVC video without sound).

  On success, the sent Message is returned. Bots can currently send animation files of up to 50 MB in size, this limit may be changed in the future.

  |          Parameter          |                                       Type                                       | Required |                                                                                                                                                                                                                                                                     Description                                                                                                                                                                                                                                                                    |
  |:---------------------------:|:--------------------------------------------------------------------------------:|:--------:|:--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------:|
  | chat_id                     | Integer or String                                                                | Yes      | Unique identifier for the target chat or username of the target channel (in the format @channelusername)                                                                                                                                                                                                                                                                                                                                                                                                                                           |
  | message_thread_id           | Integer                                                                          | Optional | Unique identifier for the target message thread (topic) of the forum; for forum supergroups only                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
  | animation                   | InputFile or String                                                              | Yes      | Animation  to send. Pass a file_id as String to send an animation that exists on  the Telegram servers (recommended), pass an HTTP URL as a String for  Telegram to get an animation from the Internet, or upload a new  animation using multipart/form-data. More information on Sending Files »                                                                                                                                                                                                                                                  |
  | duration                    | Integer                                                                          | Optional | Duration of sent animation in seconds                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
  | width                       | Integer                                                                          | Optional | Animation width                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
  | height                      | Integer                                                                          | Optional | Animation height                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
  | thumbnail                   | InputFile or String                                                              | Optional | Thumbnail  of the file sent; can be ignored if thumbnail generation for the file  is supported server-side. The thumbnail should be in JPEG format and  less than 200 kB in size. A thumbnail's width and height should not  exceed 320. Ignored if the file is not uploaded using  multipart/form-data. Thumbnails can't be reused and can be only uploaded  as a new file, so you can pass “attach://<file_attach_name>” if  the thumbnail was uploaded using multipart/form-data under  <file_attach_name>. More information on Sending Files » |
  | caption                     | String                                                                           | Optional | Animation caption (may also be used when resending animation by file_id), 0-1024 characters after entities parsing                                                                                                                                                                                                                                                                                                                                                                                                                                 |
  | parse_mode                  | String                                                                           | Optional | Mode for parsing entities in the animation caption. See formatting options for more details.                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
  | caption_entities            | Array of MessageEntity                                                           | Optional | A JSON-serialized list of special entities that appear in the caption, which can be specified instead of parse_mode                                                                                                                                                                                                                                                                                                                                                                                                                                |
  | has_spoiler                 | Boolean                                                                          | Optional | Pass True if the animation needs to be covered with a spoiler animation                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
  | disable_notification        | Boolean                                                                          | Optional | Sends the message silently. Users will receive a notification with no sound.                                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
  | protect_content             | Boolean                                                                          | Optional | Protects the contents of the sent message from forwarding and saving                                                                                                                                                                                                                                                                                                                                                                                                                                                                               |
  | reply_to_message_id         | Integer                                                                          | Optional | If the message is a reply, ID of the original message                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
  | allow_sending_without_reply | Boolean                                                                          | Optional | Pass True if the message should be sent even if the specified replied-to message is not found                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
  | reply_markup                | InlineKeyboardMarkup or ReplyKeyboardMarkup or ReplyKeyboardRemove or ForceReply | Optional | Additional interface options. A JSON-serialized object for an inline keyboard, custom reply keyboard, instructions to remove reply keyboard or to force a reply from the user.                                                                                                                                                                                                                                                                                                                                                                     |"
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token payload ^File animation]
  (c/telegram-request-upload token payload animation :animation))


(defn send-voice
  "Use this method to send audio files, if you want Telegram clients to display the file as a playable voice message.

  For this to work, your audio must be in an .OGG file encoded with OPUS (other formats may be sent as Audio or Document).
  On success, the sent Message is returned. Bots can currently send voice messages of up to 50 MB in size, this limit may be changed in the future.

  |          Parameter          |                                       Type                                       | Required |                                                                                                                                    Description                                                                                                                                   |
  |:---------------------------:|:--------------------------------------------------------------------------------:|:--------:|:--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------:|
  | chat_id                     | Integer or String                                                                | Yes      | Unique identifier for the target chat or username of the target channel (in the format @channelusername)                                                                                                                                                                         |
  | message_thread_id           | Integer                                                                          | Optional | Unique identifier for the target message thread (topic) of the forum; for forum supergroups only                                                                                                                                                                                 |
  | voice                       | InputFile or String                                                              | Yes      | Audio  file to send. Pass a file_id as String to send a file that exists on  the Telegram servers (recommended), pass an HTTP URL as a String for  Telegram to get a file from the Internet, or upload a new one using  multipart/form-data. More information on Sending Files » |
  | caption                     | String                                                                           | Optional | Voice message caption, 0-1024 characters after entities parsing                                                                                                                                                                                                                  |
  | parse_mode                  | String                                                                           | Optional | Mode for parsing entities in the voice message caption. See formatting options for more details.                                                                                                                                                                                 |
  | caption_entities            | Array of MessageEntity                                                           | Optional | A JSON-serialized list of special entities that appear in the caption, which can be specified instead of parse_mode                                                                                                                                                              |
  | duration                    | Integer                                                                          | Optional | Duration of the voice message in seconds                                                                                                                                                                                                                                         |
  | disable_notification        | Boolean                                                                          | Optional | Sends the message silently. Users will receive a notification with no sound.                                                                                                                                                                                                     |
  | protect_content             | Boolean                                                                          | Optional | Protects the contents of the sent message from forwarding and saving                                                                                                                                                                                                             |
  | reply_to_message_id         | Integer                                                                          | Optional | If the message is a reply, ID of the original message                                                                                                                                                                                                                            |
  | allow_sending_without_reply | Boolean                                                                          | Optional | Pass True if the message should be sent even if the specified replied-to message is not found                                                                                                                                                                                    |
  | reply_markup                | InlineKeyboardMarkup or ReplyKeyboardMarkup or ReplyKeyboardRemove or ForceReply | Optional | Additional interface options. A JSON-serialized object for an inline keyboard, custom reply keyboard, instructions to remove reply keyboard or to force a reply from the user.                                                                                                   |"
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token payload ^File voice]
  (c/telegram-request-upload token payload voice :voice))


(defn send-video-note
  "As of v.4.0, Telegram clients support rounded square MPEG4 videos of up to 1 minute long.
  Use this method to send video messages. On success, the sent Message is returned.

  |          Parameter          |                                       Type                                       | Required |                                                                                                                                                                                                                                                                     Description                                                                                                                                                                                                                                                                    |
  |:---------------------------:|:--------------------------------------------------------------------------------:|:--------:|:--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------:|
  | chat_id                     | Integer or String                                                                | Yes      | Unique identifier for the target chat or username of the target channel (in the format @channelusername)                                                                                                                                                                                                                                                                                                                                                                                                                                           |
  | message_thread_id           | Integer                                                                          | Optional | Unique identifier for the target message thread (topic) of the forum; for forum supergroups only                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
  | video_note                  | InputFile or String                                                              | Yes      | Video  note to send. Pass a file_id as String to send a video note that exists  on the Telegram servers (recommended) or upload a new video using  multipart/form-data. More information on Sending Files ». Sending video notes by a URL is currently unsupported                                                                                                                                                                                                                                                                                 |
  | duration                    | Integer                                                                          | Optional | Duration of sent video in seconds                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
  | length                      | Integer                                                                          | Optional | Video width and height, i.e. diameter of the video message                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
  | thumbnail                   | InputFile or String                                                              | Optional | Thumbnail  of the file sent; can be ignored if thumbnail generation for the file  is supported server-side. The thumbnail should be in JPEG format and  less than 200 kB in size. A thumbnail's width and height should not  exceed 320. Ignored if the file is not uploaded using  multipart/form-data. Thumbnails can't be reused and can be only uploaded  as a new file, so you can pass “attach://<file_attach_name>” if  the thumbnail was uploaded using multipart/form-data under  <file_attach_name>. More information on Sending Files » |
  | disable_notification        | Boolean                                                                          | Optional | Sends the message silently. Users will receive a notification with no sound.                                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
  | protect_content             | Boolean                                                                          | Optional | Protects the contents of the sent message from forwarding and saving                                                                                                                                                                                                                                                                                                                                                                                                                                                                               |
  | reply_to_message_id         | Integer                                                                          | Optional | If the message is a reply, ID of the original message                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
  | allow_sending_without_reply | Boolean                                                                          | Optional | Pass True if the message should be sent even if the specified replied-to message is not found                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
  | reply_markup                | InlineKeyboardMarkup or ReplyKeyboardMarkup or ReplyKeyboardRemove or ForceReply | Optional | Additional interface options. A JSON-serialized object for an inline keyboard, custom reply keyboard, instructions to remove reply keyboard or to force a reply from the user.                                                                                                                                                                                                                                                                                                                                                                     |"
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token payload ^File note]
  (c/telegram-request-upload token payload note :videoNote))


(defn send-media-group [])
(defn send-location [])
(defn send-venue [])
(defn send-contact [])
(defn send-poll [])
(defn send-dice [])
(defn send-chat-action [])

(defn get-user-profile-photos [])
(defn get-file [])

(defn ban-chat-member [])
(defn unban-chat-member [])
(defn restrict-chat-member [])
(defn promote-chat-member [])
(defn set-chat-administrator-custom-title [])
(defn ban-chat-sender-chat [])
(defn unban-chat-sender-chat [])

(defn set-chat-permissions [])
(defn export-chat-invite-link [])
(defn create-chat-invite-link [])
(defn edit-chat-invite-link [])
(defn revoke-chat-invite-link [])
(defn approve-chat-join-request [])
(defn decline-chat-join-request [])
(defn set-chat-photo [])
(defn delete-chat-photo [])
(defn set-chat-title [])
(defn set-chat-description [])
(defn pin-chat-message [])
(defn unpin-chat-message [])
(defn unpin-all-chat-messages [])
(defn leave-chat [])
(defn get-chat [])
(defn get-chat-administrators [])
(defn get-chat-member-count [])
(defn set-chat-sticker-set [])
(defn delete-chat-sticker-set [])
(defn get-forum-topic-icon-stickers [])
(defn create-forum-topic [])
(defn edit-forum-topic [])
(defn close-forum-topic [])
(defn reopen-forum-topic [])
(defn delete-forum-topic [])
(defn unpin-all-forum-topic-messages [])
(defn edit-general-forum-topic [])
(defn close-general-forum-topic [])
(defn reopen-general-forum-topic [])
(defn hide-general-forum-topic [])
(defn unhide-general-forum-topic [])
(defn unpin-all-general-forum-topic-messages [])
(defn answer-callback-query [])
(defn set-my-commands [])
(defn delete-my-commands [])
(defn get-my-commands [])
(defn set-my-name [])
(defn get-my-name [])
(defn set-my-description [])
(defn set-my-short-description [])
(defn get-my-short-description [])
(defn set-chat-menu-button [])
(defn get-chat-menu-button [])
(defn set-my-default-administrator-rights [])
(defn get-my-default-administrator-rights [])
(defn edit-message-text [])
(defn edit-message-caption [])
(defn edit-message-media [])
(defn edit-message-live-location [])
(defn stop-message-live-location [])
(defn edit-message-reply-markup [])
(defn stop-poll [])
(defn delete-message [])

(defn send-sticker [])
(defn get-sticker-set [])
(defn get-custom-emoji-stickers [])
(defn upload-sticker-file [])
(defn create-new-sticker-set [])
(defn add-sticker-to-set [])
(defn set-sticker-position-in-set [])
(defn delete-sticker-from-set [])
(defn set-sticker-emoji-list [])
(defn set-sticker-keywords [])
(defn set-sticker-mask-position [])
(defn set-sticker-set-title [])
(defn set-sticker-set-thumbnail [])
(defn set-custom-sticker-set-thumbnail [])
(defn delete-sticker-set [])

(defn answer-inline-query [])
(defn answer-webapp-query [])

(defn send-invoice [])
(defn create-invoice-link [])
(defn answer-shipping-query [])
(defn answer-precheckout-query [])

(defn send-game [])
(defn set-game-score [])
(defn get-game-high-scores [])
