(ns heyoka.client
  (:require [clj-http.client :as h]
            [jsonista.core :as json]
            [heyoka.utils :refer [kebabize-key]]))

(def ^:private api-endpoint
  "API endpoint used for HTTP calls"
  "https://api.telegram.org/bot")


(def object-mapper
  (json/object-mapper {:encode-key-fn name
                       :decode-key-fn (comp keyword kebabize-key)}))

(defn- build-request-url
  "Build url for request, with token"
  {:added "0.1.0"}
  [method ^String token]
  (str api-endpoint token "/" (name method)))


(defn- generate-telegram-error
  [{:keys [description error-code] :as full} payload]
  (merge (condp = error-code
           404 {:telegram/not-found description}

           {:telegram/unknown-error description})
         {:response full :payload payload}))


(defn telegram-request
  "High level API request method"
  {:added "0.1.0"}
  ([token method] (telegram-request token method {}))
  ([token method params] (telegram-request token method params nil))
  ([token method params multipart-data]
   (let [target (build-request-url method token)
         {:keys [body code]}
         (h/get target
                (cond-> {:content-type     :json
                         :debug true
                         :throw-exceptions false}
                  (seq params)    (assoc :query-string params)
                  multipart-data  (assoc :multipart multipart-data)))
         {:keys [ok result] :as full} (json/read-value body object-mapper)]
     (if ok result
         (throw (ex-info "Telegram API Call Exception"
                         (generate-telegram-error full params)))))))
