(ns heyoka.api
  (:require [heyoka.client :as c]
            [schema.core :as s]
            [heyoka.models :refer :all]))


(defn whoami
  "A simple method for testing your bot's authentication token.
  Requires no parameters.

  Returns basic information about the bot in form of a User object."
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token]
  (c/telegram-request token :getMe))


(defn logout
  "Use this method to log out from the cloud Bot API server before launching the bot locally.

  You must log out the bot before running it locally, otherwise there is no
  guarantee that the bot will receive updates. After a successful call,
  you can immediately log in on a local server, but will not be able to log in
  back to the cloud Bot API server for 10 minutes.

  Returns True on success. Requires no parameters."
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token]
  (c/telegram-request token :logOut))


(defn close
  "Use this method to close the bot instance before moving it from one local
   server to another.

  You need to delete the webhook before calling this method to ensure that the
  bot isn't launched again after server restart.
  The method will return error 429 in the first 10 minutes after the bot is
  launched.

  Returns True on success. Requires no parameters."
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token]
  (c/telegram-request token :close))


(defn send-message
  "Use this method to send text messages.

  On success, the sent Message is returned.


  Parameters:

  |          Parameter          |                                       Type                                       | Required |                                                                                   Description                                                                                  |
  |:---------------------------:|:--------------------------------------------------------------------------------:|:--------:|:------------------------------------------------------------------------------------------------------------------------------------------------------------------------------:|
  | chat_id                     | Integer or String                                                                | Yes      | Unique identifier for the target chat or username of the target channel (in the format @channelusername)                                                                       |
  | message_thread_id           | Integer                                                                          | Optional | Unique identifier for the target message thread (topic) of the forum; for forum supergroups only                                                                               |
  | text                        | String                                                                           | Yes      | Text of the message to be sent, 1-4096 characters after entities parsing                                                                                                       |
  | parse_mode                  | String                                                                           | Optional | Mode for parsing entities in the message text. See formatting options for more details.                                                                                        |
  | entities                    | Array of MessageEntity                                                           | Optional | A JSON-serialized list of special entities that appear in message text, which can be specified instead of parse_mode                                                           |
  | disable_web_page_preview    | Boolean                                                                          | Optional | Disables link previews for links in this message                                                                                                                               |
  | disable_notification        | Boolean                                                                          | Optional | Sends the message silently. Users will receive a notification with no sound.                                                                                                   |
  | protect_content             | Boolean                                                                          | Optional | Protects the contents of the sent message from forwarding and saving                                                                                                           |
  | reply_to_message_id         | Integer                                                                          | Optional | If the message is a reply, ID of the original message                                                                                                                          |
  | allow_sending_without_reply | Boolean                                                                          | Optional | Pass True if the message should be sent even if the specified replied-to message is not found                                                                                  |
  | reply_markup                | InlineKeyboardMarkup or ReplyKeyboardMarkup or ReplyKeyboardRemove or ForceReply | Optional | Additional interface options. A JSON-serialized object for an inline keyboard, custom reply keyboard, instructions to remove reply keyboard or to force a reply from the user. |

  ** Formatting options

  The Bot API supports basic formatting for messages. You can use bold, italic,
  underlined, strikethrough, and spoiler text, as well as inline links and pre-formatted code in your bots' messages.
  Telegram clients will render them accordingly.
  You can specify text entities directly, or use markdown-style or HTML-style formatting.

  Note that Telegram clients will display an alert to the user before opening an inline link
  ('Open this link?' together with the full URL).

  Message entities can be nested, providing following restrictions are met:
  - If two entities have common characters, then one of them is fully contained inside another.
  - bold, italic, underline, strikethrough, and spoiler entities can contain and can be part of any other entities, except pre and code.
  - All other entities can't contain each other.

  Links tg://user?id=<user_id> can be used to mention a user by their ID without using a username. Please note:

    These links will work only if they are used inside an inline link or in an inline keyboard button. For example, they will not work, when used in a message text.
    Unless the user is a member of the chat where they were mentioned, these mentions are only guaranteed to work if the user has contacted the bot in private in the past or has sent a callback query to the bot via an inline button and doesn't have Forwarded Messages privacy enabled for the bot."
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token payload]
  (s/validate MessagePayload payload)
  (c/telegram-request token :sendMessage
                      payload))


(defn forward-message
  "Use this method to forward messages of any kind.

  Service messages can't be forwarded.

  On success, the sent Message is returned.

  |       Parameter      |        Type       | Required |                                                       Description                                                       |
  |:--------------------:|:-----------------:|:--------:|:-----------------------------------------------------------------------------------------------------------------------:|
  | chat_id              | Integer or String | Yes      | Unique identifier for the target chat or username of the target channel (in the format @channelusername)                |
  | message_thread_id    | Integer           | Optional | Unique identifier for the target message thread (topic) of the forum; for forum supergroups only                        |
  | from_chat_id         | Integer or String | Yes      | Unique identifier for the chat where the original message was sent (or channel username in the format @channelusername) |
  | disable_notification | Boolean           | Optional | Sends the message silently. Users will receive a notification with no sound.                                            |
  | protect_content      | Boolean           | Optional | Protects the contents of the forwarded message from forwarding and saving                                               |
  | message_id           | Integer           | Yes      | Message identifier in the chat specified in from_chat_id                                                                |"
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token payload]
  (s/validate MessageForwardPayload payload)
  (c/telegram-request token :forwardMessage
                      payload))


(defn copy-message
  "Use this method to copy messages of any kind.

  Service messages and invoice messages can't be copied.
  A quiz poll can be copied only if the value of the field correct_option_id is
  known to the bot. The method is analogous to the method forwardMessage, but
  the copied message doesn't have a link to the original message.

  Returns the MessageId of the sent message on success.


  |          Parameter          |                                       Type                                       | Required |                                                                                   Description                                                                                  |
  |:---------------------------:|:--------------------------------------------------------------------------------:|:--------:|:------------------------------------------------------------------------------------------------------------------------------------------------------------------------------:|
  | chat_id                     | Integer or String                                                                | Yes      | Unique identifier for the target chat or username of the target channel (in the format @channelusername)                                                                       |
  | message_thread_id           | Integer                                                                          | Optional | Unique identifier for the target message thread (topic) of the forum; for forum supergroups only                                                                               |
  | from_chat_id                | Integer or String                                                                | Yes      | Unique identifier for the chat where the original message was sent (or channel username in the format @channelusername)                                                        |
  | message_id                  | Integer                                                                          | Yes      | Message identifier in the chat specified in from_chat_id                                                                                                                       |
  | caption                     | String                                                                           | Optional | New caption for media, 0-1024 characters after entities parsing. If not specified, the original caption is kept                                                                |
  | parse_mode                  | String                                                                           | Optional | Mode for parsing entities in the new caption. See formatting options for more details.                                                                                         |
  | caption_entities            | Array of MessageEntity                                                           | Optional | A JSON-serialized list of special entities that appear in the new caption, which can be specified instead of parse_mode                                                        |
  | disable_notification        | Boolean                                                                          | Optional | Sends the message silently. Users will receive a notification with no sound.                                                                                                   |
  | protect_content             | Boolean                                                                          | Optional | Protects the contents of the sent message from forwarding and saving                                                                                                           |
  | reply_to_message_id         | Integer                                                                          | Optional | If the message is a reply, ID of the original message                                                                                                                          |
  | allow_sending_without_reply | Boolean                                                                          | Optional | Pass True if the message should be sent even if the specified replied-to message is not found                                                                                  |
  | reply_markup                | InlineKeyboardMarkup or ReplyKeyboardMarkup or ReplyKeyboardRemove or ForceReply | Optional | Additional interface options. A JSON-serialized object for an inline keyboard, custom reply keyboard, instructions to remove reply keyboard or to force a reply from the user. |"
  {:added "0.1.0" :tg-api-version "0.6.9"}
  [token payload]
  (s/validate MessageCopyPayload payload)
  (c/telegram-request token :copyMessage payload))
