(ns geospatial.geocoding
  "Geocoding utils based on externals API's"
  (:require [clj-http.client :as http]
            [charred.api :as json]
            [geospatial.geojson.specs :refer [assert-geojson!]]
            [geospatial.codecs :as codecs]
            [geospatial.codecs.core :as codec]
            [geospatial.geojson :as geojson]))

(defprotocol GeocodingService
  (-search [this] "High-level search method")
  (reverse-search [this] "High-level reverse-search method"))

;;; Geoplateforme Endpoint

(defonce geoplateforme-endpoint "https://data.geopf.fr/geocodage")
(defrecord GeoplateformeGouvFr [query]
  GeocodingService
  (-search [{:keys [query]}]
    (when query
      (let [batched? (sequential? query)
            results (-> (http/get (format "%s/search" geoplateforme-endpoint)
                                  {:query-params {:q query}})
                        :body
                        (json/read-json :key-fn keyword)
                        (dissoc :query)
                        ;; Ugly hack to reverse the cood positions since service returns
                        ;; are incorrects
                        (geojson/transpose-geojson-coordinates reverse))]
        (assert-geojson! results)))))

;;; Google Geocoding API

(defonce google-api-endpoint "https://maps.googleapis.com/maps/api/geocode/json")
(defrecord GoogleAPIGeocoding [api-key query]
  GeocodingService
  (-search [{:keys [query api-key]}]
    (let [key? (System/getenv "GOOGLE_GEOCODING_API_KEY")]
      (when query
        (let [{:keys [results status]}
              (-> (http/get google-api-endpoint
                            {:query-params
                             (cond-> {:address query}
                               api-key (assoc :key api-key)
                               key? (assoc :key key?))})
                  :body
                  (json/read-json :key-fn keyword))]
          (when (= status "OK")
            (->> results
                 (codecs/convert codecs/GoogleGeocoding codecs/GeoJSONCodec)
                 (assert-geojson!))))))))

(def default-geocode-params {:service :geoplateforme :codec :geojson})
(defn geocode
  "Retrieve geocoding of query.

   Params:
     `service` Backend service to use, default use `Geoplateforme`
     `codec` Codec to use to encode results, default to `GeoJSON`"
  {:added "0.1.0"}
  ([query]
   (geocode query default-geocode-params))
  ([query {:keys [service codec api-key]}]
   (when-let [builder (condp = service
                        :geoplateforme ->GeoplateformeGouvFr
                        :google        (partial ->GoogleAPIGeocoding api-key)
                        (throw
                         (ex-info "No Geocoding implementation found"
                                  {:service service
                                   :query   query})))]
     (->> (builder query)
          -search))))
