(ns geospatial.codecs.formats.googlegeocoding
  (:require [geospatial.codecs.core :as codec]
            [geospatial.codecs.models :as models]
            [geospatial.codecs.formats]
            [geospatial.geojson :as geojson]
            [malli.core :as m]
            [clojure.string :as str])
  (:import geospatial.codecs.formats.GoogleGeocoding))

(defn- encode-google-geocode-result*
  [{:keys [formatted_address geometry address_components]}]
  {:entity/geometry
   {:geometry/type :point
    :geometry/coordinates [(get-in geometry [:location :lat])
                           (get-in geometry [:location :lng])]}
   :entity/attributes
   (merge
    (reduce (fn [acc {:keys [long_name types]}]
              (assoc acc (keyword (str/join "|" types))
                     long_name))
            {} address_components)
    {:address formatted_address})})

(extend-type GoogleGeocoding
  codec/GeoCodec

  (-decode [_ data]
    (let [results (mapv encode-google-geocode-result* data)]
      {:doc/entities results}))

  (-encode [_ data]
    (m/assert models/Document data)
    (->>
     (:doc/entities data)
     (map (fn [entity]
            {:address_components
             (->> (get-in entity [:entity/attributes])
                  (map (fn [[type long_name]]
                         {:long_name long_name
                          :types (str/split (name type) #"\|")})))
             :formatted_address (get-in entity [:entity/attributes :address])
             :geometry
             {:location {:lat (get-in entity [:entity/geometry :geometry/coordinates 0])
                         :lng (get-in entity [:entity/geometry :geometry/coordinates 1])}
              :location_type "ROOFTOP"}
             :types ["street_address" "subpremise"]})))))
