(ns geospatial.geojson.specs
  "GeoJSON RFC-7946 compliant spec"
  (:require [malli.core :as m]
            [malli.error :as me]))

;;   ________                  ____.
;;  /  _____/  ____  ____     |    | __________   ____
;; /   \  ____/ __ \/  _ \    |    |/  ___/  _ \ /    \
;; \    \_\  \  ___(  <_> )\__|    |\___ (  <_> )   |  \
;;  \______  /\___  >____/\________/____  >____/|___|  /
;;         \/     \/                    \/           \/

(def AxisPrimitive
  "A primitive value representing a single axis in GeoJSON,
  such as longitude or easting."
  [:double {:description "Longitude / Easting"
            :min -180 :max 180}])

(def Id
  "An identifier for a GeoJSON Geometry object, which may be
  a string or integer, as specified in the GeoJSON RFC."
  [:or {:description "GeoJSON Geometry ID"}
   :string :int])

(def Position
  "A position in GeoJSON, defined as an array of numbers representing
  longitude, latitude, and optionally altitude, per the GeoJSON RFC."
  [:vector {:description "A single position: [longitude, latitude] or [longitude, latitude, altitude]."
            :examples [[100.0 0.0] [100.0 0.0 10.0]]
            :min 2 :max 3}
   [:ref :geojson.primitives/axis]])

(def LinearRing
  "A closed LineString with four or more positions, where the first
  and last positions are equivalent, as required by the GeoJSON RFC for Polygon boundaries."
  [:and
   [:vector {:min 4}
    [:ref :geojson/position]]
   [:fn {:description "A linear ring must have at least 4 positions and the first and last must be identical."
         :examples [[100.0 0.0] [101.0 0.0] [101.0 1.0] [100.0 0.0]]}
    (fn [ring]
      (and (= (first ring) (last ring))
           (>= (count ring) 4)))]])

(def Bbox
  "A bounding box array representing the minimum and maximum extents of a geometry.

   4 for 2D plan & 6 for 3D Plan"
  [:vector {:min 4 :max 6}
   [:ref :geojson.primitives/axis]])

(def Point
  "A GeoJSON Point geometry object, representing a single
  position in coordinate space, as specified in the GeoJSON RFC."
  [:map
   {:description "A GeoJSON Point geometry."
    :examples [{:type "Point" :coordinates [100.0 0.0]}]}
   [:type [:enum "Point"]]
   [:bbox {:optional true}
    [:ref :geojson.primitives/bbox]]
   [:coordinates [:ref :geojson/position]]])

(def MultiPoint
  "A GeoJSON MultiPoint geometry object, representing an array
  of Point positions."
  [:map
   {:description "A GeoJSON MultiPoint geometry."
    :examples [{:type "MultiPoint" :coordinates [[100.0 0.0] [101.0 1.0]]}]}
   [:type [:enum "MultiPoint"]]
   [:bbox {:optional true}
    [:ref :geojson.primitives/bbox]]
   [:coordinates [:vector [:ref :geojson/position]]]])

(def LineString
  "A GeoJSON LineString geometry object, representing a sequence of
  two or more positions, as described in the GeoJSON RFC."
  [:map
   {:description "A GeoJSON LineString geometry."
    :examples [{:type "LineString" :coordinates [[100.0 0.0] [101.0 1.0]]}]}
   [:type [:enum "LineString"]]
   [:bbox {:optional true}
    [:ref :geojson.primitives/bbox]]
   [:coordinates [:vector {:min 2} [:ref :geojson/position]]]])

(def MultiLineString
  "A GeoJSON MultiLineString geometry object, representing an array
  of LineString coordinate arrays, as specified in the GeoJSON RFC."
  [:map
   {:description "A GeoJSON MultiLineString geometry."
    :examples [{:type "MultiLineString" :coordinates [[[100.0 0.0] [101.0 1.0]] [[102.0 2.0] [103.0 3.0]]]}]}
   [:type [:enum "MultiLineString"]]
   [:bbox {:optional true}
    [:ref :geojson.primitives/bbox]]
   [:coordinates [:vector [:vector {:min 2} [:ref :geojson/position]]]]])

(def Polygon
  "A GeoJSON Polygon geometry object, representing an array
  of LinearRing coordinate arrays, as defined in the GeoJSON RFC."
  [:map
   {:description "A GeoJSON Polygon geometry."
    :examples [{:type "Polygon"
                :coordinates [[[100.0 0.0] [101.0 0.0] [101.0 1.0] [100.0 0.0]]]}]}
   [:type [:enum "Polygon"]]
   [:bbox {:optional true}
    [:ref :geojson.primitives/bbox]]
   [:coordinates [:vector [:ref :geojson/linear-ring]]]])

(def MultiPolygon
  "A GeoJSON MultiPolygon geometry object, representing an
   array of Polygon coordinate arrays, as described in the GeoJSON RFC."
  [:map
   {:description "A GeoJSON MultiPolygon geometry."
    :examples [{:type "MultiPolygon"
                :coordinates [[[[100.0 0.0] [101.0 0.0] [101.0 1.0] [100.0 0.0]]]]}]}
   [:type [:enum "MultiPolygon"]]
   [:bbox {:optional true}
    [:ref :geojson.primitives/bbox]]
   [:coordinates [:vector [:vector [:ref :geojson/linear-ring]]]]])

(def GeometryCollection
  "A GeoJSON GeometryCollection object, containing an array
  of geometry objects, as specified by the GeoJSON RFC."
  [:map
   {:description "A geometry collection (recursive)."
    :examples [{:type "GeometryCollection"
                :geometries [{:type "Point" :coordinates [100.0 0.0]}]}]}
   [:type [:enum "GeometryCollection"]]
   [:bbox {:optional true}
    [:ref :geojson.primitives/bbox]]
   [:geometries [:vector [:ref :geojson/geometry]]]])

(def Geometry
  "A generic GeoJSON geometry object, which may be a Point, MultiPoint,
  LineString, MultiLineString, Polygon, MultiPolygon, or GeometryCollection, as defined in the GeoJSON RFC."
  [:or {:description "Any GeoJSON geometry object."}
   [:ref :geojson/point]
   [:ref :geojson/multi-point]
   [:ref :geojson/linestring]
   [:ref :geojson/multi-linestring]
   [:ref :geojson/polygon]
   [:ref :geojson/multi-polygon]
   [:ref :geojson/geometry-collection]])

(def Properties
  "A GeoJSON properties object, representing a map of
  arbitrary name-value pairs, as described in the GeoJSON RFC."
  [:map {:description "Geometry Properties"
         :examples [{:foo "bar"}]}])

(def Feature
  "A GeoJSON Feature object, which includes geometry and
  associated properties, as specified in the GeoJSON RFC."
  [:map
   {:description "A GeoJSON Feature object."
    :examples [{:type "Feature"
                :geometry {:type "Point" :coordinates [100.0 0.0]}
                :properties {:name "Example"}}]}
   [:type [:enum "Feature"]]
   [:geometry [:maybe [:ref :geojson/geometry]]]
   [:bbox {:optional true}
    [:ref :geojson.primitives/bbox]]
   [:properties [:ref :geojson/properties]]
   [:id {:optional true}
    [:ref :geojson.primitives/id]]])

(def FeatureCollection
  "A GeoJSON FeatureCollection object, representing an array of Feature objects, as defined in the GeoJSON RFC."
  [:map
   {:description "A GeoJSON FeatureCollection object."
    :examples [{:type "FeatureCollection"
                :features [{:type "Feature"
                            :geometry {:type "Point" :coordinates [100.0 0.0]}
                            :properties {:name "Example"}}]}]}
   [:type [:enum "FeatureCollection"]]
   [:bbox {:optional true}
    [:ref :geojson.primitives/bbox]]
   [:properties {:optional true}
    [:ref :geojson/properties]]
   [:id {:optional true}
    [:ref :geojson.primitives/id]]
   [:features [:vector [:ref :geojson/feature]]]])

(def GeoJSON
  "A top-level GeoJSON object, which may be a Feature,
  FeatureCollection, or Geometry, as specified in the GeoJSON RFC."
  [:or
   {:description "Any valid GeoJSON object (Feature, FeatureCollection, or Geometry)."
    :examples [{:type "FeatureCollection" :features []}
               {:type "Feature" :geometry nil :properties {}}
               {:type "Point" :coordinates [100.0 0.0]}]}
   [:ref :geojson/geometry]
   [:ref :geojson/feature]
   [:ref :geojson/feature-collection]])

;;; ====================================================================

(def ^:private registry
  "A registry mapping GeoJSON schema keywords to their definitions,
    as required for schema validation."
  {:geojson.primitives/axis AxisPrimitive
   :geojson.primitives/bbox Bbox
   :geojson.primitives/id Id
   :geojson/position Position
   :geojson/linear-ring LinearRing
   :geojson/properties Properties
   :geojson/point Point
   :geojson/multi-point MultiPoint
   :geojson/linestring LineString
   :geojson/multi-linestring MultiLineString
   :geojson/polygon Polygon
   :geojson/multi-polygon MultiPolygon
   :geojson/geometry-collection GeometryCollection
   :geojson/geometry Geometry
   :geojson/feature Feature
   :geojson/feature-collection FeatureCollection
   ::geojson GeoJSON})

(def geojson-schema
  "The default schema for validating GeoJSON object"
  (m/schema ::geojson {:registry (merge (m/default-schemas) registry)}))

(def valid-geojson?
  "A validation function that checks if a value conforms to the GeoJSON schema."
  (m/validator geojson-schema))
