(ns component.webserver
  "Webserver component"
  (:require
   [integrant.core :as ig]
   [clojure.tools.logging :as log]
   [io.pedestal.http :as http]
   [reitit
    [http :as rhttp]
    [pedestal :as rpedestal]]
   [reitit.coercion :refer [encode-error]]
   [reitit.http.coercion :as coercion]
   [reitit.http.interceptors
    [parameters :as ri.parameters]
    [muuntaja :as ri.muuntaja]]
   [reitit.coercion.malli :as coercion-malli]
   [reitit.interceptor]
   [reitit.http.interceptors.dev :refer [print-context-diffs]]
   [muuntaja.core :as muuntaja]
   [jsonista.core :as json]
   [lib.webserver.routes :as routes]
   [lib.webserver.utils :as wsutils]
   [lib.webserver.interceptors :as witcp]))

;; ___________         __                        __
;; \_   _____/__  ____/  |_____________    _____/  |_
;;  |    __)_\  \/  /\   __\_  __ \__  \ _/ ___\   __\
;;  |        \>    <  |  |  |  | \// __ \\  \___|  |
;; /_______  /__/\_ \ |__|  |__|  (____  /\___  >__|
;;         \/      \/                  \/     \/

(defmethod ig/init-key :component.webserver/route
  [_ {:keys [routes] :as config}]
  (if (fn? routes)
    (routes (dissoc config :routes))
    routes))

(defn register-routing
  "Convenient wrapper to inject routes symbols as
   refset entities"
  {:added "1.1.0"}
  [routes]
  (let [[routes-var cfg] (if (var? routes)
                           [routes nil]
                           routes)]
    {[:component.webserver/route (keyword (symbol routes-var))]
     (assoc cfg :routes (var-get routes-var))}))

(defmethod ig/init-key :component.webserver/resources    [_ resources] resources)
(defmethod ig/init-key :component.webserver/interceptors [_ itcps] itcps)

;; ___________                                .___
;; \_   _____/__  ______________    ____    __| _/
;;  |    __)_\  \/  /\____ \__  \  /    \  / __ |
;;  |        \>    < |  |_> > __ \|   |  \/ /_/ |
;; /_______  /__/\_ \|   __(____  /___|  /\____ |
;;         \/      \/|__|       \/     \/      \/

(defmethod ig/expand-key :component/webserver
  [k {:keys [env http servlet default-routes-config resources interceptors malli-options] :as v}]
  {k (merge v
            {:env     (or env :dev)
             :servlet (or servlet false)
             :http (merge #::http{:join? false
                                  :secure-headers nil
                                  :resource-path "/public"
                                  :type :jetty
                                  :host "0.0.0.0"
                                  :port 8080
                                  :routes []
                                  :tracing nil}
                          http)
             :default-routes-config default-routes-config
             :routes        (ig/refset :component.webserver/route)
             :resources     (or resources (ig/ref :component.webserver/resources))
             :interceptors  (or interceptors (ig/refset :component.webserver/interceptors))
             :malli-options (or malli-options
                                {:error-keys #{:type :humanized :schema}
                                 :strip-extra-keys true
                                 :default-values true})
             :muuntaja-formatter (muuntaja/create muuntaja/default-options)})})
;; .___  __
;; |   |/  |_  ____ ______
;; |   \   __\/ ___\\____ \
;; |   ||  | \  \___|  |_> >
;; |___||__|  \___  >   __/
;;                \/|__|

(defn resource-itcp
  "Inject resources conresponding the reitit spec"
  {:added "1.1.0"}
  [id {:keys [itcp] :as resource}]
  (or itcp
      {:name (keyword (str "add-" (name id)))
       :enter (fn [ctx] (assoc-in ctx [:request id] resource))}))

(defn check-resources
  "Verify  ressource availability and add corresponding itcps.

   If not available remove route (todo option to throw)"
  {:added "1.1.0"}
  [[p m :as route] {:keys [resources] :as ops}]
  (if-let [required (:resources m)]
    (when (every? resources required)
      (rhttp/coerce-handler
       [p
        (update m :interceptors
                (partial into (mapv (fn [resource-id]
                                      (resource-itcp resource-id
                                                     (resources resource-id)))
                                    required)))]
       ops))
    (rhttp/coerce-handler route ops)))

(defn exceptions-interceptors
  "Default pretty format exception catcher"
  []
  (letfn [(dispatch-error [ex]
            (let [data (ex-data ex)]
              (condp = (:type data)

                :muuntaja/decode
                {:status 400
                 :headers {"Content-Type" "application/json"}
                 :body (json/write-value-as-string {:status 400
                                                    :message "Content Negociation Issue"
                                                    :error (encode-error data)})}

                ;; Input Request Coercion Error
                ;; If the user send data not conform to the current route
                :reitit.coercion/request-coercion
                {:status 400
                 :headers {"Content-Type" "application/json"}
                 :body {:status 400
                        :message "Request Coercion Error"
                        :error (select-keys (encode-error data) #{:humanized})}}

                ;; Output Response Coercion Error
                ;; If the server respond data that is not conform to the current route
                :reitit.coercion/response-coercion
                {:status 400
                 :headers {"Content-Type" "application/json"}
                 :body (json/write-value-as-string {:status 500
                                                    :message "Result Coercion Error"
                                                    :error (select-keys (encode-error data) #{:humanized})})}

                ;; Default Handler
                ;; If catched, return the stuff here
                {:status 500
                 :headers {"Content-Type" "application/json"}
                 :body (json/write-value-as-string
                        (cond-> {:status 500
                                 :message "Internal Server Error"}
                          (some? data) (assoc :error (encode-error data))))})))]
    {:name ::catch-exceptions
     :error (fn [ctx ^Throwable ex]
              (try
                (let [response (dispatch-error ex)]
                  (assoc ctx :response response))
                (catch Exception e
                  (log/error e)
                  ;; Most likely will never happen, but catch to be sure
                  ;; and safe
                  (assoc ctx :response
                         {:status 500
                          :body (json/write-value-as-string
                                 {:status 500
                                  :message "Internal Server Error"})}))))}))

(defn- mk-interceptor-stack
  "Build the default stack of interceptors"
  [{:keys [_metrics _sentry _telemetry sentry]}]
  [(witcp/compiled-sentry-interceptor sentry)
   (exceptions-interceptors)
   (ri.parameters/parameters-interceptor)
   (ri.muuntaja/format-interceptor)
   (coercion/coerce-response-interceptor)
   (coercion/coerce-request-interceptor)])

(defn default-interceptors
  "1. Flatten to support \"sub\" interceptors stack
   2. Remove skipped interceptors based on dependency
      availability status"
  [resources]
  (->> (mk-interceptor-stack resources)
       (flatten)
       (keep identity)
       (vec)))
;; .___       .__  __
;; |   | ____ |__|/  |_
;; |   |/    \|  \   __\
;; |   |   |  \  ||  |
;; |___|___|  /__||__|
;;          \/

(defmethod ig/init-key :component/webserver
  [_ {:keys [http env debug servlet routes interceptors
             resources muuntaja-formatter malli-options]}]
  (let [http-config http
        router (rhttp/router
                [(routes/stock-routes resources) routes]
                (cond-> {:coerce check-resources
                         :resources resources
                         :data {:muuntaja muuntaja-formatter
                                :coercion (coercion-malli/create malli-options)
                                :interceptors (if (seq interceptors)
                                                (into [] interceptors)
                                                (default-interceptors resources))}}
                  ;; Context Diff on dev
                  (and (= env :dev) debug) (assoc :reitit.interceptor/transform print-context-diffs)))
        service-map (-> http-config
                        (http/default-interceptors)
                        (rpedestal/replace-last-interceptor
                         (rpedestal/routing-interceptor router))
                        ;; Allow CORS & Exceptions in dev context
                        (cond-> (= env :dev) (http/dev-interceptors)))
        server (if-not servlet
                 (let [serv (http/create-server service-map)]
                   (log/info "starting" (name env) "web server on"
                             (str (::http/host http-config) ":" (::http/port http-config))
                             "...")
                   (http/start serv))
                 (do
                   (log/info "starting applet" (name env) "web server...")
                   (http/create-servlet service-map)))]
    (when debug
      (wsutils/pprint-ring-routes router))
    {:server server}))

;;   ___ ___        .__   __
;;  /   |   \_____  |  |_/  |_
;; /    ~    \__  \ |  |\   __\
;; \    Y    // __ \|  |_|  |
;;  \___|_  /(____  /____/__|
;;        \/      \/

(defmethod ig/halt-key! :component/webserver
  [_ {:keys [server]}]
  (try
    (log/info "stopping web server")
    (when server
      (http/stop server))
    (catch Exception e
      (log/errorf e "error stopping pedestal server"))))
