(ns component.webserver
  (:require [integrant.core :as ig]
            [commons.system :refer [assert-system!]]
            [commons.runtime :refer [runtime-properties]]
            [muuntaja.core :as muuntaja]
            [io.pedestal.http :as http]
            [reitit
             [http     :as rhttp]
             [pedestal :as rpedestal]]
            [reitit.http.coercion :as coercion]
            [reitit.http.interceptors
             [parameters :as ri.parameters]
             [muuntaja   :as ri.muuntaja]]
            [schema.core :as s]
            [clojure.spec.alpha :as sa]
            [reitit.http.interceptors.dev :refer [print-context-diffs]]
            [reitit.swagger :as swagger]
            [reitit.swagger-ui :as swagger-ui]
            [clojure.tools.logging :as log]))

(def ^:dynamic *telemetry-available?*
  (try
    (require '[lib.telemetry.interceptors :as ti])
    true
    (catch Exception _
      false)))

;;  __      __      ___.     _________
;; /  \    /  \ ____\_ |__  /   _____/ ______________  __ ___________
;; \   \/\/   // __ \| __ \ \_____  \_/ __ \_  __ \  \/ // __ \_  __ \
;;  \        /\  ___/| \_\ \/        \  ___/|  | \/\   /\  ___/|  | \/
;;   \__/\  /  \___  >___  /_______  /\___  >__|    \_/  \___  >__|
;;        \/       \/    \/        \/     \/                 \/

(def ^:private route-aggregator-key    :component.webserver/route)
(def ^:private resource-aggregator-key :component.webserver/resources)

;;; Register Funcs

(defn- register-routing
  "generate derived route that will be aggregated
   into the webserver prep stage"
  [routes]
  (let [[routes-var cfg] (if (var? routes)
                           [routes nil]
                           routes)
        rk (keyword (symbol routes-var))]
    (log/trace "registered route" rk)
    {[route-aggregator-key rk]
     (assoc cfg :routes (var-get routes-var))}))

(defn register-all-routes
  "merge all route into one component"
  [& all-routes]
  (reduce merge (for [routes all-routes]
                  (register-routing routes))))

(defmethod ig/init-key route-aggregator-key
  [_ {:keys [routes] :as config}]
  (if (fn? routes)
    (routes (dissoc config :routes))
    routes))

(defmethod ig/init-key resource-aggregator-key
  [_ resources] resources)

(defn register-resources
  "public facade for resource registering
  use a map with integrant reference as value

  eq:

  {:logging (ig/ref :component/logging)
   :sentry  (ig/ref :component/sentry)}
  "
  [resource-map]
  {resource-aggregator-key resource-map})

;;; Default Routes

(defn- default-ping-handler
  [app-infos resources]
  (let [pingable (filter (comp :ping-fn second) resources)]
    (fn [_]
      (let [pings (->> pingable
                       (map (fn [[k {ping :ping-fn}]]
                              (try
                                (let [start (System/currentTimeMillis)
                                      r (ping)
                                      delay (- (System/currentTimeMillis) start)]
                                  [k (assoc r :delay delay)])
                                (catch Exception e
                                  (log/error "ping" k (.getMessage e))
                                  {k {:ok false
                                      :message (.getMessage e)}}))))
                       (into {}))
            ok? (every? (comp :ok val) pings)]
        (log/trace "ping call handled")
        {:status (if ok? 200 599)
         :body
         (merge app-infos
                {:ok         ok?
                 :hostname   (.getHostName (java.net.InetAddress/getLocalHost))
                 :resources  (keys resources)
                 :components pings})}))))

(defn default-ping-route
  [{:keys [enable-ping?] :as _system}
   {:keys [app-infos] :as resources}]
  (when enable-ping?
    ["/ping" {:name ::ping
              :no-metrics true
              :tags ["monitoring"]
              :resources []
              :get {:summary "route for health check"
                    :description
                    "health check route returning
                        + `ok`         boolean indicating if everything works fine
                        + `version`    current running version of the app
                        + `app`        app name
                        + `hostname`   ...
                        + `components` a map component statuses containing at least a boolean `ok`"
                    :parameters {}
                    :handler (default-ping-handler app-infos resources)
                    :responses {200 {:body {:ok s/Bool
                                            :version s/Any
                                            :app s/Any
                                            :hostname s/Str}}}}}]))

(defn- ->swagger
  [{:keys [app-infos]}]
  (when app-infos
    {:info {:title (str (:name app-infos) " API")
            :version (:version app-infos)}}))

(defn- default-swagger-routes
  [{:keys [disabled? info]}]
  ["" {:no-doc     true
       :no-metrics true}
   ["/swagger.json" {:get {:swagger {:info info}
                           :handler (swagger/create-swagger-handler)}}]
   ["/swagger/*"    {:get (swagger-ui/create-swagger-ui-handler)}]])

(defn- default-metrics-interceptor
  [{:keys [registry scrape-fn]}]
  {:name ::prometheus-metrics-scrape
   :enter
   (fn [ctx]
     (assoc ctx :response
            {:status 200
             :headers {"Content-Type" "text/plain"}
             :body (.getBytes ^String (scrape-fn registry))}))})

(defn- default-metrics-route
  [metrics {:keys [enable-metrics?]}]
  (when (and enable-metrics? metrics)
    ["/metrics" {:get {:no-doc true
                       :resources [:metrics]
                       :name ::get-metrics
                       :interceptors [(default-metrics-interceptor metrics)]}}]))

(defn- default-webserver-routes
  [conf {:keys [metrics] :as resources}]
  [(default-ping-route     conf resources)
   (default-swagger-routes (merge (->swagger resources) resources))
   (default-metrics-route metrics conf)])

;;; Helpers

(defn- resource-itcp
  "create resource interceptor"
  [id {:keys [itcp] :as resource}]
  (or itcp
      {:name (keyword (str "add-" (name id)))
       :enter (fn [ctx] (assoc-in ctx [:request id] resource))}))

(defn- check-resources
  "Verify ressource availability and add corresponding itcps
   if not available remove route (todo option to throw)"
  [[p m :as route] {:keys [resources] :as ops}]
  (if-let [required (:resources m)]
    (when (every? resources required)
      (rhttp/coerce-handler
       [p
        (update m :interceptors
                (partial concat (map (fn [resource-id]
                                       (resource-itcp resource-id
                                                      (resources resource-id)))
                                     required)))]
       ops))
    (rhttp/coerce-handler route ops)))

(defn default-interceptors-stack
  [_resources]
  [(ri.parameters/parameters-interceptor)
   (ri.muuntaja/format-interceptor)
   (ri.muuntaja/format-response-interceptor)
   (coercion/coerce-response-interceptor)
   (coercion/coerce-request-interceptor)])

;;; Integrant

(sa/def :component.webserver/env keyword?)

(defmethod ig/assert-key :component/webserver [_ system]
  (assert (sa/valid? (sa/keys :req-un [:component.webserver/env])
                     system)))

(defmethod ig/prep-key :component/webserver
  [_ {:keys [env http servlet default-routes routes resources]
      :as config}]
  (merge config {:env (or env :dev)
                 :servlet (or servlet false)
                 :http (merge #::http{:join? false
                                      :secure-headers nil
                                      :resource-path "/public"
                                      :type :jetty
                                      :host "0.0.0.0"
                                      :port 8080
                                      :routes []}
                              http)
                 :default-routes default-routes
                 :routes (or routes (ig/refset :component.webserver/route))
                 :resources (or resources (ig/refset :component.webserver/resource))
                 :enable-swagger? true
                 :enable-ping?    true
                 :enable-metrics? true
                 :content-negociator
                 (muuntaja/create
                  (assoc-in
                   muuntaja/default-options
                   [:formats "application/json" :encoder-opts]
                   {:date-format (or (:date-format http) "yyyy-MM-dd'T'HH:mm:ss.SSSX")}))}))

(defmethod ig/init-key :component/webserver
  [_ {:keys [http env debug? routes resources interceptors content-negociator] :as system}]
  (let [pedestal-conf (merge #::http{:join?          false
                                     :secure-headers nil
                                     :resource-path  "/public"
                                     :type           :jetty
                                     :host           "0.0.0.0"
                                     :port           8080
                                     :routes         []}
                             http)
        configuration (-> pedestal-conf
                          (http/default-interceptors)
                          (rpedestal/replace-last-interceptor
                           (rpedestal/routing-interceptor
                            (rhttp/router
                             [(default-webserver-routes system resources)
                              routes]
                             (let [telemetry-injected? (contains? (into #{} (keys resources))
                                                                  :telemetry)]
                               (cond-> {:coerce    check-resources
                                        :resources resources
                                        :data      {:muuntaja content-negociator
                                                    :interceptors (if interceptors
                                                                    (into [] interceptors)
                                                                    (default-interceptors-stack resources))}}
                                 (and (= env :dev) debug?)
                                 (assoc :reitit.interceptor/transform print-context-diffs)

                                 ;; Hack warning, runtime resolve telemetry interceptors,
                                 ;; and inject them into the stack
                                 (and telemetry-injected? *telemetry-available?*)
                                 (update-in [:data :interceptors]
                                            #(concat ((eval 'ti/telemetry-interceptors)
                                                      (:telemetry resources)))))))))
                          (cond-> (= env :dev) (http/dev-interceptors)))
        server (let [serv (http/create-server configuration)]
                 (log/info "starting" (name env) "web server on"
                           (str (::http/host configuration)
                                ":" (::http/port configuration)))
                 (http/start serv))
        ping-fn (fn [_server]
                  ;; TODO: do something to check the health status of
                  ;; the server instance
                  {:ok true})]
    (assert-system!
     (assoc system :instance server
            :ping-fn (partial ping-fn server)
            :props-fn runtime-properties))))

(defmethod ig/halt-key! :component/webserver
  [_ {:keys [instance]}]
  (when instance
    (log/info "stopping webserver")
    (http/stop instance)))
