(ns component.telemetry
  (:require [integrant.core :as ig]
            [malli.core :as m]
            [clojure.tools.logging :as log]
            [medley.core :refer [deep-merge]]
            [steffan-westcott.clj-otel.sdk.otel-sdk :as sdk]
            [steffan-westcott.clj-otel.api.trace.http :as trace-http]
            [steffan-westcott.clj-otel.exporter.otlp.http.trace :as ht]
            [steffan-westcott.clj-otel.api.trace.span :as span]
            [steffan-westcott.clj-otel.context :as context])
  (:gen-class))

(defn ^:private mk-tracer-provider
  "Initialize the tracing provider"
  [collector-endpoint sampler]
  {:added "1.0.0"}
  (cond-> {:tracer-provider {:span-processors
                             [{:exporters [(ht/span-exporter {:endpoint collector-endpoint})]}]}}
    sampler (assoc-in [:tracer-provider :sampler] sampler)))

(def ^:private bounded-interceptors
  "Default interceptor stack exposed to the component, mean to
   be used jointly with the webserver component, automatic injection
   will be done if the `tracing` resource is present."
  {:pre  (trace-http/server-span-interceptors {:create-span? true})
   :post [(trace-http/route-interceptor)]})

;; ___________                                .___
;; \_   _____/__  ______________    ____    __| _/
;;  |    __)_\  \/  /\____ \__  \  /    \  / __ |
;;  |        \>    < |  |_> > __ \|   |  \/ /_/ |
;; /_______  /__/\_ \|   __(____  /___|  /\____ |
;;         \/      \/|__|       \/     \/      \/

(defmethod ig/expand-key :component/telemetry
  [k v]
  {k (deep-merge
      {:enabled? true
       :sampler {:always :on
                 :ratio 0.2
                 :parent-based {:root {:always :on}}}
       :collector-endpoint "http://localhost:4318/v1/traces"}
      v)})

;;    _____                               __
;;   /  _  \   ______ ______ ____________/  |_
;;  /  /_\  \ /  ___//  ___// __ \_  __ \   __\
;; /    |    \\___ \ \___ \\  ___/|  | \/|  |
;; \____|__  /____  >____  >\___  >__|   |__|
;;         \/     \/     \/     \/

(def spec
  [:map
   [:app-name {:title "Application Name"
               :description "The application name"
               :json-schema/example "acme"}
    string?
    :collector-endpoint {:title "Collector Endpoint"
                         :description "Url of Opentelemetry Collector"
                         :json-schema/example "http://localhost:4318/v1/traces"}]])

(defmethod ig/assert-key :component/telemetry
  [_ system]
  (assert (m/validate spec system)
          (m/explain spec system)))

;; .___       .__  __
;; |   | ____ |__|/  |_
;; |   |/    \|  \   __\
;; |   |   |  \  ||  |
;; |___|___|  /__||__|
;;          \/

(defmethod ig/init-key :component/telemetry
  [_ {:keys [enabled? app-name collector-endpoint sampler] :as sys}]
  (if enabled?
    (let [resources  []
          spec (cond-> {}
                 true      (merge (mk-tracer-provider collector-endpoint sampler))
                 resources (assoc :resources resources))
          sdk (sdk/init-otel-sdk! app-name spec)]
      (log/infof "starting telemetry component on collector endpoint %s with sampling %s"
                 collector-endpoint (:ratio sampler))
      (sdk/add-shutdown-hook! sdk)
      (assoc sys
             :sdk sdk
             :interceptors bounded-interceptors
             :context-extractor context/->headers
             :generic-instrument-fn (fn instrumented-fn [args body-fn]
                                      (span/with-span! args
                                        (body-fn)))))
    (log/info "telemetry component started but disabled")))

;;   ___ ___        .__   __
;;  /   |   \_____  |  |_/  |_
;; /    ~    \__  \ |  |\   __\
;; \    Y    // __ \|  |_|  |
;;  \___|_  /(____  /____/__|
;;        \/      \/

(defmethod ig/halt-key! :component/telemetry
  [_ {:keys [sdk] :as sys}]
  (log/info "stopping telemetry component")
  (when sdk
    (sdk/close-otel-sdk! sdk))
  sys)
