(ns component.telemetry
  (:require [integrant.core :as ig]
            [clojure.spec.alpha :as s]
            [commons.system :refer [assert-system!]]
            [commons.runtime :refer [runtime-properties]]
            [clojure.tools.logging :as log])
  (:import io.opentelemetry.sdk.OpenTelemetrySdk
           io.opentelemetry.api.GlobalOpenTelemetry
           io.opentelemetry.sdk.trace.SdkTracerProvider
           io.opentelemetry.api.trace.propagation.W3CTraceContextPropagator
           io.opentelemetry.api.baggage.propagation.W3CBaggagePropagator
           io.opentelemetry.context.propagation.ContextPropagators
           io.opentelemetry.context.propagation.TextMapPropagator
           io.opentelemetry.exporter.otlp.trace.OtlpGrpcSpanExporter
           io.opentelemetry.exporter.otlp.http.trace.OtlpHttpSpanExporter
           io.opentelemetry.sdk.trace.export.BatchSpanProcessor
           io.opentelemetry.sdk.trace.export.SimpleSpanProcessor
           io.opentelemetry.semconv.ResourceAttributes
           io.opentelemetry.sdk.resources.Resource))

;; ___________    .__                         __
;; \__    ___/___ |  |   ____   _____   _____/  |________ ___.__.
;;   |    |_/ __ \|  | _/ __ \ /     \_/ __ \   __\_  __ <   |  |
;;   |    |\  ___/|  |_\  ___/|  Y Y  \  ___/|  |  |  | \/\___  |
;;   |____| \___  >____/\___  >__|_|  /\___  >__|  |__|   / ____|
;;              \/          \/      \/     \/             \/

;;; Config

(def ^:private default-backend-conf
  {:mode :grpc
   :endpoint (or (System/getenv "OTEL_EXPORTER_OTLP_ENDPOINT")
                 "http://127.0.0.1:4318")})

(def default-configuration
  {:component/telemetry {:logging (ig/ref :component/logging)
                         :attributes {}
                         :batched? true
                         :backends {:trace   default-backend-conf
                                    :metrics default-backend-conf
                                    :logs    default-backend-conf}}})

;;; Spec

(s/def ::service-name string?)
(s/def ::batched? boolean?)
(s/def ::endpoint string?)
(s/def ::mode #{:http :grpc})
(s/def ::backends (s/map-of #{:trace :metrics :logs}
                            (s/keys :req-un [::mode
                                             ::endpoint])))
(s/def ::component
  (s/keys :req-un [::service-name
                   ::backends
                   ::batched?]))

(defmethod ig/assert-key :component/telemetry [_ system]
  (assert (s/valid? ::component system)
          (s/explain-str ::component system)))

;;; Component

(defn- ^Resource otel-build-resources
  "Configure the entity producing telemetry as resource attributes."
  [{:keys [service-name attributes]}]
  (let [runtime-props (merge (runtime-properties) attributes)
        r (cond-> (.toBuilder (Resource/getDefault))
            service-name (.put ResourceAttributes/SERVICE_NAME service-name))]
    (.build r)))

(defn- ^SdkTracerProvider otel-build-tracer
  "Build Tracer Instance"
  [resources {:keys [backends batched?] :as _system}]
  (let [{:keys [mode endpoint]} (:trace backends)
        span-exporter (condp = mode
                        :http (-> (OtlpHttpSpanExporter/builder)
                                  (.setEndpoint endpoint)
                                  (.build))
                        :grpc (-> (OtlpGrpcSpanExporter/builder)
                                  (.setEndpoint endpoint)
                                  (.build)))
        batch-processor (if batched?
                          (.build (BatchSpanProcessor/builder span-exporter))
                          (.build (SimpleSpanProcessor/builder span-exporter)))
        sdk (-> (SdkTracerProvider/builder)
                (.addSpanProcessor batch-processor)
                (.setResource resources))]
    (.build sdk)))

(defn- ^ContextPropagators otel-build-propagators
  [_system]
  (ContextPropagators/create
   (TextMapPropagator/composite
    [(W3CTraceContextPropagator/getInstance)
     (W3CBaggagePropagator/getInstance)])))

;;; Integrant Keys

(defmethod ig/init-key :component/telemetry
  [_ {:keys [backends] :as system}]
  (let [res (otel-build-resources system)
        tracer (otel-build-tracer res system)
        propagators (otel-build-propagators system)
        otlp (-> (OpenTelemetrySdk/builder)
                 (.setTracerProvider tracer)
                 (.setPropagators propagators))
        otlp-sdk (.buildAndRegisterGlobal otlp)]
    (log/info "starting telemetry component")
    (assert-system!
     (assoc system
            :sdk otlp-sdk
            :ping-fn (constantly (merge backends {:ok true}))
            :props-fn runtime-properties))))

(defmethod ig/suspend-key! :component/telemetry [_ _]
  (log/info "unregistred telemetry global context")
  (GlobalOpenTelemetry/resetForTest))

(defmethod ig/halt-key! :component/telemetry
  [_ {:keys [sdk] :as system}]
  (try
    (when sdk
      (log/info "shutting down sdk")
      (.shutdown sdk))
    (catch Exception e
      (log/warn "issue while un-registering SDK %s" (.getMessage e))
      (GlobalOpenTelemetry/resetForTest)))
  (dissoc system :sdk))
