(ns lib.sentry.core
  (:require [lib.sentry.api :as sa]
            [lib.sentry.exceptions :as se]
            [clojure.tools.logging :as log]
            [medley.core :as m]))

(defn http-data
  "Append Http information as `sentry.interfaces.Http` data
  @see https://docs.sentry.io/development/sdk-dev/interfaces/#http-interface"
  [{:keys [url method headers query_string data env]}]
  {"sentry.interfaces.Http"
   (cond-> {}
     url (assoc :url url)
     method (assoc :method method)
     headers (assoc :headers headers)
     query_string (assoc :query_string query_string)
     data (assoc :data data)
     env (assoc :env env))})

(defn http-pedestal-data
  "Build http-data from pedestal request"
  [{:keys [request]}]
  (http-data
   {:url          (str (name (:scheme request))
                       "://"
                       (:server-name request)
                       (when (not= 80 (:server-port request))
                         (str ":" (:server-port request)))
                       (:uri request))
    :method       (:request-method request)
    :headers      (get request :headers {})
    :query_string (get request :query-string "")
    :data         (:body-params request {})
    :env          {:remote_addr (get request :remote-addr "")}}))

(defn exception-data
  "Exception data can be added to the data in capture"
  [ex]
  (-> {}
      (se/message ex)
      (se/tags ex)
      (se/stacktrace ex)))

(defn capture
  "sentry-event a map that may contain message, level or extra
   e.g. {:message 'Ooops'
        :level :warning}
   with level, one of: :debug, :info, :warning, :error, :fatal"
  [dsn sentry-event & data]
  (try
    (sa/capture dsn (merge (apply m/deep-merge :extra data) sentry-event))
    (catch Exception ex
      (log/error ex (str "could not send exception to sentry :" (.getMessage ex))))))

;;; Public API

(defn send-exception
  [dsn ex ctx data _logger]
  (capture dsn
           data
           (exception-data ex)
           (http-pedestal-data ctx)))
